<?php declare(strict_types = 1);

namespace EshopStock\AdminModule\Components;

use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Templating\Filters\Price as PriceFilter;
use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use Core\Model\UI\DataGrid\DataSource\DoctrineDataSource;
use Doctrine\ORM\Query\Expr\Join;
use Doctrine\ORM\QueryBuilder;
use EshopCatalog\Model\Entities\Product;
use EshopCatalog\Model\Entities\ProductListener;
use EshopCatalog\Model\Entities\ProductTexts;
use EshopCatalog\Model\Entities\Supplier;
use EshopStock\AdminModule\Presenters\DefaultPresenter;
use EshopStock\Model\Entities\Stock;
use EshopStock\Model\Entities\Supply;
use EshopStock\Model\EshopStockConfig;
use EshopStock\Model\Repository\Stocks;
use EshopStock\Model\Repository\Supplies;
use EshopStock\Model\Repository\SupplyProducts;
use Nette\Application\UI\Multiplier;
use Nette\Security\User;

/**
 * @property-read DefaultPresenter $presenter
 */
class OverviewGrid extends BaseControl
{
	protected Stock        $stock;
	protected QueryBuilder $queryBuilder;

	public function __construct(
		int                            $stockId,
		protected Stocks               $stocksService,
		protected Supplies             $supplies,
		protected PriceFilter          $priceFilter,
		protected SupplyProducts       $supplyProducts,
		protected ISupplyDetailFactory $supplyDetailFactory,
		protected User                 $user
	)
	{
		$this->stock                              = $this->stocksService->get($stockId);
		EntityManagerDecorator::$useDoctrineQuery = true;
		$this->queryBuilder                       = $this->getQueryBuilder();
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	protected function getQueryBuilder(): QueryBuilder
	{
		$qb = $this->supplies->getQueryBuilder($this->stock->getId());
		$qb->addOrderBy('s.dateSupply', 'desc');

		return $qb;
	}

	public function createComponentGrid(): BaseDataGrid
	{
		$grid                    = $this->createGrid();
		$grid->detailEyePosition = (string) EshopStockConfig::loadScalar('detailEyePosition') ?: 'right';

		$grid->setAutoSubmit(false);
		$grid->setDataSource($this->queryBuilder);

		/** @var array<int, string> $suppliersNames */
		$suppliersNames = [];

		/** @var DoctrineDataSource $dataSource */
		$dataSource                 = $grid->getDataSource();
		$dataSource->onDataLoaded[] = function(array $items) use (&$suppliersNames): void {
			/** @var Supply[] $items */
			$ids         = [];
			$supplierIds = [];

			foreach ($items as $item) {
				$ids[$item->getId()]                   = $item->getId();
				$supplierIds[$item->supplier->getId()] = $item->supplier->getId();
			}

			if ($supplierIds !== []) {
				foreach ($this->em->getRepository(Supplier::class)->createQueryBuilder('s')
					         ->select('s.id, s.name')
					         ->where('s.id IN (' . implode(',', $supplierIds) . ')')
					         ->getQuery()->getArrayResult() as $row) {
					$suppliersNames[(int) $row['id']] = (string) $row['name'];
				}
			}
		};

		$grid->setOuterFilterRendering();
		$grid->addFilterText('search', 'eshopStock.overviewGrid.search')
			->setCondition(static function(\Core\Model\Entities\QueryBuilder $qb, $value): void {
				$qb->innerJoin('s.supplier', 'supp')
					->andWhere('s.invoiceNumber LIKE :q OR supp.name LIKE :q')
					->setParameter('q', "%{$value}%");
			});
		$grid->addFilterText('searchProduct', 'eshopStock.overviewGrid.searchProduct')
			->setCondition(function(\Core\Model\Entities\QueryBuilder $qb, $value): void {
				$subQProd = $this->em->getRepository(Product::class)->createQueryBuilder('p')
					->select('p.id')
					->where('p.ean LIKE :pq OR p.code1 LIKE :pq OR p.code2 LIKE :pq');
				$qb->andWhere('sp.product IN (' . $subQProd->getQuery()->getDQL() . ')')
					->setParameter('pq', "%{$value}%");
			});
		$grid->addFilterText('searchProductName', 'eshopStock.overviewGrid.searchProductName')
			->setCondition(function(\Core\Model\Entities\QueryBuilder $qb, $value): void {
				$subQText = $this->em->getRepository(ProductTexts::class)->createQueryBuilder('pt')
					->select('IDENTITY(pt.id)')
					->where('pt.name LIKE :pq');
				$qb->andWhere('sp.product IN (' . $subQText->getQuery()->getDQL() . ')')
					->setParameter('pq', "%{$value}%");
			});

		$grid->addColumnDateTime('dateSupply', 'eshopStock.overviewGrid.date')
			->setAlign('left');
		$grid->addColumnText('supplier', 'eshopStock.overviewGrid.supplier')
			->setRenderer(static function(Supply $supply) use (&$suppliersNames): string {
				return $suppliersNames[$supply->supplier->getId()] ?: '';
			});
		$grid->addColumnText('invoiceNumber', 'eshopStock.overviewGrid.invoiceNumber');
		$grid->addColumnText('purchasePriceSum', 'eshopStock.overviewGrid.price')
			->setRenderer(fn(Supply $supply): string => $this->priceFilter->format($supply->getPurchasePriceSum()));
		$grid->setItemsDetail(__DIR__ . '/OverviewGridDetail.latte');
		$grid->addAction('edit', '', 'removeSupply!')
			->setRenderCondition(fn(): bool => $this->isAllowedEdit())
			->addClass('ajax')
			->setConfirm('default.reallyDelete')
			->setIcon('trash')
			->setBsType('danger');

		return $grid;
	}

	public function createComponentSupplyDetail(): Multiplier
	{
		return new Multiplier(function($supplyId) {
			$qb = $this->queryBuilder;
			$qb->addSelect('p, pt, m, vr')
				->andWhere('s.id = :supply')->setParameter('supply', $supplyId)
				->innerJoin('sp.product', 'p')
				->innerJoin('p.productTexts', 'pt', Join::WITH, 'pt.lang = :lang')
				->leftJoin('p.manufacturer', 'm')
				->leftJoin('p.vatRate', 'vr')
				->setParameter('lang', $this->translator->getLocale());

			return $this->supplyDetailFactory->create($qb->getQuery()->getOneOrNullResult());
		});
	}

	/**
	 * @param int|string $id
	 */
	public function handleRemoveSupply($id): void
	{
		if (!$this->isAllowedEdit()) {
			return;
		}
		ProductListener::$updateVariant = false;

		if ($this->supplies->removeSupply((int) $id)) {
			$this->presenter->flashMessageSuccess('default.removed');
			$this['grid']->reload();
		} else {
			$this->presenter->flashMessageDanger('default.removeFailed');
		}

		$this->presenter->redrawControl('flashes');
	}

	public function isAllowedEdit(): bool
	{
		return $this->user->isAllowed($this->presenter->getFullModuleName(), 'edit');
	}

}
