<?php declare(strict_types = 1);

namespace EshopStock\Model\Entities;

use Core\Model\Entities\TId;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\DBAL\Types\Types;
use Doctrine\ORM\Mapping as ORM;
use EshopCatalog\Model\Entities\Product;
use EshopOrders\Model\Entities\Order;
use EshopStock\Model\Listeners\ProductToOrderOrderListener;
use Nette\InvalidArgumentException;

#[ORM\Table('eshop_stock__product_to_order')]
#[ORM\Entity]
#[ORM\EntityListeners([ProductToOrderOrderListener::class])]
class ProductToOrder
{
	use TId;

	#[ORM\JoinColumn(name: 'product_id', referencedColumnName: 'id', onDelete: 'CASCADE')]
	#[ORM\ManyToOne(targetEntity: Product::class)]
	public ?Product $product = null;

	/** @var Collection<ProductsToOrderOrders> */
	#[ORM\OneToMany(mappedBy: 'productToOrder', targetEntity: ProductsToOrderOrders::class, cascade: ['all'])]
	public Collection $orders;

	#[ORM\Column(type: Types::INTEGER)]
	public int $quantity;

	/**
	 * @param Order[] $orders
	 */
	public function __construct(Product $product, array $orders, int $quantity)
	{
		$this->orders = new ArrayCollection;
		$this->setQuantity($quantity);
		$this->setOrders($orders);
		$this->product  = $product;
		$this->quantity = $quantity;
	}

	public function setQuantity(int $quantity): void
	{
		if ($quantity >= 0) {
			throw new InvalidArgumentException('$quantity must be negative number');
		}
		$this->quantity = $quantity;
	}

	/**
	 * @param Order[] $orders
	 */
	public function setOrders(array $orders): void
	{
		/** @phpstan-ignore-next-line */
		if ($this->orders === null) {
			$this->orders = new ArrayCollection;
		}

		if (count($orders) === 0) {
			throw new InvalidArgumentException('$orders must have at least one item');
		}

		foreach ($orders as $order) {
			$this->addOrder($order);
		}
	}

	public function addOrder(Order $order): void
	{
		$this->orders->add(new ProductsToOrderOrders($order, $this));
	}

}
