<?php declare(strict_types = 1);

namespace EshopStock\Model\Repository;

use Closure;
use Contributte\Translation\Translator;
use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Entities\EntityRepository;
use DateTimeInterface;
use Doctrine\ORM\Query\Expr\Join;
use Doctrine\ORM\QueryBuilder;
use EshopStock\Model\Entities\Supply;
use EshopStock\Model\Entities\SupplyProduct;

class SupplyItemQuery
{
	/** @var Closure[] */
	protected array $filter = [];

	/** @var Closure[] */
	protected array              $orderBy = [];
	protected ?DateTimeInterface $toDate  = null;

	public function __construct(
		protected EntityManagerDecorator $em,
		protected Translator             $translator,
		protected int                    $stockId,
	)
	{
	}

	public function withSupplier(int $supplier): self
	{
		$this->filter[] = static function(QueryBuilder $queryBuilder, QueryBuilder $subQueryBuilder) use ($supplier): void {
			$queryBuilder->andWhere($queryBuilder->expr()->eq('supp1', $supplier));
			$subQueryBuilder->andWhere($queryBuilder->expr()->eq('supp2', $supplier));
		};

		return $this;
	}

	public function withManufacturer(int $manufacturer): self
	{
		$this->filter[] = static function(QueryBuilder $queryBuilder, QueryBuilder $subQueryBuilder) use ($manufacturer): void {
			$queryBuilder->andWhere($queryBuilder->expr()->eq('m1', $manufacturer));
			$subQueryBuilder->andWhere($queryBuilder->expr()->eq('m2', $manufacturer));
		};

		return $this;
	}

	public function toDate(DateTimeInterface $dateTime): self
	{
		$this->toDate   = $dateTime;
		$this->filter[] = static function(QueryBuilder $queryBuilder, QueryBuilder $subQueryBuilder) use ($dateTime): void {
			$queryBuilder->andWhere($queryBuilder->expr()->lte('s1.dateSupply', '\'' . sprintf('%s %s', $dateTime->format('Y-m-d'), '23:59:59') . '\''));
		};

		return $this;
	}

	public function addOrderByCode(string $order = 'asc'): self
	{
		$this->orderBy[] = static function(QueryBuilder $queryBuilder, QueryBuilder $subQueryBuilder) use ($order): void {
			$queryBuilder->addOrderBy('p1.code1', $order);
		};

		return $this;
	}

	public function addOrderByDateSupply(string $order = 'asc'): self
	{
		$this->orderBy[] = static function(QueryBuilder $queryBuilder, QueryBuilder $subQueryBuilder) use ($order): void {
			$queryBuilder->addOrderBy('s1.dateSupply', $order);
		};

		return $this;
	}

	/**
	 * @param EntityRepository|\Doctrine\ORM\EntityRepository $repository
	 */
	protected function doCreateQuery($repository): QueryBuilder
	{
		// subquery na pocty kusu daneho produktu v dane prijemce
		$subQueryBuilder = $this->em->createQueryBuilder();
		$subQueryBuilder->select('COUNT(sp2.id)')
			->from(SupplyProduct::class, 'sp2')
			->join('sp2.supply', 's2')
			->join('sp2.product', 'p2')
			->leftJoin('s2.supplier', 'supp2')
			->leftJoin('p2.manufacturer', 'm2')
			->andWhere('s2.id = s1.id AND p2.id = p1.id')
			->andWhere($subQueryBuilder->expr()->isNull('sp2.order'));

		if ($this->toDate instanceof DateTimeInterface) {
			$strDateTime = sprintf('\'%s %s\'', $this->toDate->format('Y-m-d'), '23:59:59');
			$subQueryBuilder->andWhere(sprintf('(sp2.writtenOffDate IS NULL OR sp2.writtenOffDate > %s)', $strDateTime));
		}

		$subQueryBuilder->addGroupBy('p2.id');

		// hlavni dotaz, ktery iteruje pro kazdy jedinecny produkt v prijemce
		$qb = $this->em->createQueryBuilder();
		$qb->select('p1.code1, pt1.name, sp1.purchasePrice, s1.invoiceNumber, s1.dateSupply, supp1.name as supplier')
			->addSelect(sprintf('(%s) as count', $subQueryBuilder->getDQL()))
			->addSelect(sprintf('(%s) as purchasePriceSum', str_replace('2', '3', $subQueryBuilder->select('SUM(sp3.purchasePrice)')->getDQL())))
			->from(SupplyProduct::class, 'sp1')
			->join('sp1.product', 'p1')
			->join('p1.productTexts', 'pt1', Join::WITH, 'pt1.lang = :lang')
			->join('sp1.supply', 's1')
			->join('s1.stock', 'st1')
			->leftJoin('s1.supplier', 'supp1')
			->leftJoin('p1.manufacturer', 'm1')
			->andWhere($qb->expr()->eq('st1.id', $this->stockId))
			->andWhere($qb->expr()->isNull('sp1.order'));

		if ($this->toDate instanceof DateTimeInterface) {
			$strDateTime = sprintf('\'%s %s\'', $this->toDate->format('Y-m-d'), '23:59:59');
			$qb->andWhere(sprintf('(sp1.writtenOffDate IS NULL OR sp1.writtenOffDate > %s)', $strDateTime));
		}

		$qb->addGroupBy('s1.id')
			->addGroupBy('p1.id')
			->setParameter('lang', $this->translator->getLocale());

		foreach ($this->filter as $fn) {
			$fn($qb, $subQueryBuilder);
		}

		foreach ($this->orderBy as $fn) {
			$fn($qb, $subQueryBuilder);
		}

		return $qb;
	}

	public function getFullResult(): array
	{
		return $this->doCreateQuery($this->em->getRepository(Supply::class))->getQuery()->getArrayResult();
	}

	public function getSumSimplified(): array
	{
		$result = [
			'itemsCount' => 0,
			'itemsValue' => 0,
		];

		foreach ($this->getFullResult() as $item) {
			$result['itemsCount'] += (int) $item['count'];
			$result['itemsValue'] += (float) $item['purchasePriceSum'];
		}

		return $result;
	}

	public function getSum(): array
	{
		$qb = $this->doCreateQuery($this->em->getRepository(Supply::class));
		$qb->select('m1.id, m1.name as manufacturer, SUM(sp1.purchasePrice) as purchasePrice, COUNT(sp1.id) as count')
			->orderBy('m1.name')
			->groupBy('m1.id');

		return $qb->getQuery()->getArrayResult();
	}

	public function getSimplifiedResult(): array
	{
		$qb = $this->doCreateQuery($this->em->getRepository(Supply::class));
		$qb->select('p1.code1, pt1.name, COUNT(p1.id) as count')
			->groupBy('p1.id');

		return $qb->getQuery()->getArrayResult();
	}

}
