<?php declare(strict_types = 1);

namespace EshopStock\Model\Repository;

use Core\Model\Helpers\BaseEntityService;
use EshopCatalog\AdminModule\Model\Products;
use EshopOrders\Model\Entities\IOrderItem;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderItem;
use EshopStock\Model\Entities\ProductsToOrderOrders;
use EshopStock\Model\Entities\ProductToOrder;
use EshopStock\Model\Entities\Stock;
use EshopStock\Model\Entities\SupplyProduct;

/**
 * @method SupplyProduct|null get(int $id)
 */
class SupplyProducts extends BaseEntityService
{
	/** @var string */
	protected $entityClass = SupplyProduct::class;

	public static bool $assignOrderItemFlush = true;

	public function __construct(
		protected Stocks          $stocks,
		protected Products        $products,
		protected ProductsToOrder $productsToOrder,
	)
	{
	}

	/**
	 * @return SupplyProduct[]
	 */
	protected function getProductOnStock(int $productId, int $stockId): array
	{
		$qb = $this->em->createQueryBuilder();
		$qb->select('sp')->from($this->entityClass, 'sp')->join('sp.supply', 's')->join('s.stock', 'st')
			->join('sp.product', 'p')
			->where($qb->expr()->eq('st.id', $stockId))
			->andWhere($qb->expr()->isNull('sp.writtenOffDate')) // pouze neodepsane
			->andWhere($qb->expr()->isNull('sp.order')) // pouze ty, co nejsou prodane
			->andWhere($qb->expr()->eq('p.id', $productId)) // pouze od jednoho produktu
			->orderBy('s.dateSupply');

		return $qb->getQuery()->getResult();
	}

	/**
	 * @return SupplyProduct[]
	 */
	public function getProductsByOrder(int $orderId, int $productId, int $stockId): array
	{
		$qb = $this->em->createQueryBuilder();
		$qb->select('sp')->from($this->entityClass, 'sp')->join('sp.supply', 's')->join('s.stock', 'st')
			->join('sp.product', 'p')
			->join('sp.order', 'o')
			->where($qb->expr()->eq('st.id', $stockId))
			->andWhere($qb->expr()->eq('p.id', $productId)) // pouze od jednoho produktu
			->andWhere($qb->expr()->eq('o.id', $orderId)) // pouze od jedné objednávky
			->orderBy('s.dateSupply');

		return $qb->getQuery()->getResult();
	}

	public function assignOrder(Order $order): void
	{
		$defaultStock = $this->stocks->getDefaultStock();

		if (!$defaultStock instanceof Stock) {
			return;
		}

		$defaultStockId = $defaultStock->getId();

		$productsGifts = [];
		foreach ($order->getOrderItems() as $orderItem) {
			if ($orderItem->getGifts()) {
				$productsGifts = array_merge($productsGifts, $orderItem->getGifts()->toArray());
			}
		}
		$array = array_merge($order->getOrderItems()->toArray(), $order->getGifts()->toArray(), $productsGifts);
		foreach ($array as $orderItem) {
			$this->assignOrderItem($orderItem, $defaultStockId);
		}
	}

	public function removeOrder(Order $order): void
	{
		$defaultStock = $this->stocks->getDefaultStock();

		if (!$defaultStock instanceof Stock) {
			return;
		}

		$defaultStockId = $defaultStock->getId();

		$productsGifts = [];
		foreach ($order->getOrderItems() as $orderItem) {
			if ($orderItem->getGifts()) {
				$productsGifts = array_merge($productsGifts, $orderItem->getGifts()->toArray());
			}
		}
		$array = array_merge($order->getOrderItems()->toArray(), $order->getGifts()->toArray(), $productsGifts);
		foreach ($array as $orderItem) {
			$this->removeOrderItem($orderItem, $defaultStockId);
		}
	}

	public function assignOrderItem(IOrderItem $orderItem, ?int $stockId = null): void
	{
		$stockId ??= $this->stocks->getDefaultStock()->getId();
		$order   = $orderItem->getOrder();
		$flush   = [];

		$productId = $orderItem->getProductId();
		if ($productId === null) {
			return;
		}
		$product         = $this->products->get((int) $orderItem->getProductId());
		$quantity        = $orderItem->getQuantity();
		$productsOnStock = $this->getProductOnStock($productId, $stockId);

		// upraveni poctu ks u produktu
		$product->quantity -= $quantity;
		$this->em->persist($product);
		$flush[] = $product;

		// pocet ks jednoho produktu na sklade - pocet ks jednoho produktu k objednani
		$prodToOrder          = $this->productsToOrder->getOneByProduct($productId);
		$productsCountOnStock = count($productsOnStock) + ($prodToOrder->quantity ?? 0);

		// ulozit jen ty produkty objednavky, ktere jsou fyzicky na sklade
		for ($i = 0, $iMax = min($quantity, $productsCountOnStock); $i < $iMax; $i++) {
			/** @var SupplyProduct $productOnStock */
			$productOnStock        = $productsOnStock[$i];
			$productOnStock->order = $order;
			$this->em->persist($productOnStock);
			$flush[] = $productOnStock;
		}

		// produkty, ktere nejsou na sklade, vlozime do zbozi k objednani
		$quantityProductsToOrder = $productsCountOnStock - $quantity;

		// navysime stav chybejicich produktů, nebo založíme novou chybějící položku se zápornou kvantitou
		if ($quantityProductsToOrder < 0) {

			$productToOrder = $this->productsToOrder->getOneByProduct($productId);
			if (!$productToOrder instanceof ProductToOrder) {
				$productToOrder = new ProductToOrder($product, [$order], $quantityProductsToOrder);
			} else {
				$productToOrder->quantity = $quantityProductsToOrder;

				$finded = false;
				/** @var ProductsToOrderOrders $o */
				foreach ($productToOrder->orders->toArray() as $o) {
					if ($o->order->getId() === $order->getId()) {
						$finded = true;
						break;
					}
				}

				if (!$finded) {
					$productToOrder->addOrder($order);
				}
			}

			$this->em->persist($productToOrder);
			$flush[] = $productToOrder;
		}

		if (self::$assignOrderItemFlush) {
			$this->em->flush();
		}
	}

	public function removeOrderItem(IOrderItem $orderItem, ?int $stockId = null): void
	{
		$flush   = [];
		$stockId ??= $this->stocks->getDefaultStock()->getId();
		$order   = $orderItem->getOrder();

		$productId = $orderItem->getProductId();
		if ($productId === null) {
			return;
		}
		$product                            = $this->products->get((int) $orderItem->getProductId());
		$productToOrder                     = $this->productsToOrder->getOneByProduct($productId);
		$productsCountToOrder               = abs($productToOrder instanceof ProductToOrder ? $productToOrder->quantity : 0);
		$quantity                           = $orderItem->getQuantity();
		$productsInOrderOnStock             = $this->getProductsByOrder($order->getId(), $productId, $stockId);
		$defaultProductsCountInOrderOnStock = count($productsInOrderOnStock);
		/** @phpstan-ignore-next-line */
		$orderItemDefaultQuantity           = $orderItem->getDefaultQuantity() === null ? 0 : ($orderItem->getDefaultQuantity() - count($productsInOrderOnStock));
		$missingProductsCountInOrder        = $orderItemDefaultQuantity > 0 ? abs($orderItemDefaultQuantity > $quantity ? $quantity : $orderItemDefaultQuantity) : 0;
		$productsDefaultCountInOrderOnStock = $quantity - $missingProductsCountInOrder; // produkty na sklade v objednavce
		$productsDefaultCountInOrderOnStock = max(0, $productsDefaultCountInOrderOnStock);

		// TODO nebude jeste potřeba neco takove udelat i s produkty, ktere jdou do zbozi k objednani
		// v pripade ze modifikujeme ve formu pocet kusu, vytahneme jen ten pocet produktu na sklade ktery je prirustkem/ubytkem
		$productsInOrderOnStock      = array_slice($productsInOrderOnStock, 0, min($productsDefaultCountInOrderOnStock, $quantity));
		$productsCountInOrderOnStock = count($productsInOrderOnStock);

		// kolik je potřeba v produkt. v obj
		$a = $productsCountToOrder - ($quantity - $productsCountInOrderOnStock);

		// $b kolik jich pujde do produkt. k obj
		$b = $a === 0 ? 0 : min($a, $productsCountInOrderOnStock);

		// pocet ks, ktere jdou zpet do skladu
		$productsCountWillGoToStock = (int) max($productsCountInOrderOnStock - $b, 0);

		// cast produktu jde primo na sklad
		$productsWillGoToStock = [];
		if ($productsCountWillGoToStock > 0) {
			$productsWillGoToStock = array_slice($productsInOrderOnStock, 0, $productsCountWillGoToStock);
		}

		// pocet ks, ktere jdou do produktu k objednani k objednavkam, kde chybi
		$productsCountWillGoToProductsToOrder = (int) $b;

		// cast produktu jde do produktu k objednani
		$productsWillGoToProductsToOrder = [];
		if ($productsCountWillGoToProductsToOrder > 0) {
			$productsWillGoToProductsToOrder = array_slice($productsInOrderOnStock, $productsCountWillGoToStock, $productsCountWillGoToProductsToOrder);
		}

		// pocitadlo, kolik se jich realne uchyti ve zbozi k objednani
		$k = 0;

		if ($productToOrder instanceof ProductToOrder) {
			// odeberu objednavku od produktu k objednani pokud tam je
			/** @var ProductsToOrderOrders $o */
			foreach ($productToOrder->orders->toArray() as $o) {
				if ($o->order->getId() === $order->getId()) {
					// odebira se objednavka jen kdyz se rusi (stornuje) nebo nejsou v dane objednavce chybejici produkty po snizeni mnozstvi
					/** @phpstan-ignore-next-line */
					$removeOrderIf = $orderItem->getDefaultQuantity() === null || ($orderItem->getDefaultQuantity() - $defaultProductsCountInOrderOnStock - $quantity) <= 0;

					if ($removeOrderIf) {
						$productToOrder->orders->removeElement($o);
					}
					// odecteme chybejici kusy produktu, pokud se jedna pouze o snizeni poctu ks, tak odecitame jen ten snizovany rozdil
					/** @phpstan-ignore-next-line */
					$productToOrder->quantity += $orderItem->getDefaultQuantity() !== null ? $quantity : ($quantity - $productsCountInOrderOnStock);
					if ($removeOrderIf) {
						$this->em->remove($o);
					} else {
						$flush[] = $o;
					}
					break;
				}
			}

			$this->em->flush();

			$q = $productsCountWillGoToProductsToOrder;

			if ($q === 0) { // zadny produkt nejde do produktu k objednani
				// $productToOrder->quantity += $quantity;
			} else {

				// projdu objednavky a zjistim si, kolik ks skutecne chybi na objednavce (tj. rozdil mezi pozadovanym poctem na order item a kusy k te prochazene objednavce na sklade)
				/** @var ProductsToOrderOrders $p */
				foreach ($productToOrder->orders->toArray() as $p) {
					if ($p->order->getId() === $order->getId()) {
						continue;
					}
					if ($productsCountWillGoToProductsToOrder <= 0) {
						break;
					}
					$orderId = $p->order->getId();
					$cnt     = count($this->getProductsByOrder($orderId, $productId, $stockId));

					// najdu si order item na prislusny produkt
					$c = null;
					/** @var OrderItem $oi */
					foreach ($p->order->getOrderItems()->toArray() as $oi) {
						if ($oi->getProductId() === $productId) {
							$c = $oi->getQuantity() - $cnt; // zjistuji kolik je skutceny pocet kusu, ktere chybi
							break;
						}
					}

					if ($c !== null) {
						$q                        = (int) min($productsCountWillGoToProductsToOrder, $c);
						$productToOrder->quantity += $q; // snizime pocet ks

						// zmenime objednavky u produktu, u kterych dany produkt chybi a zaroven vyprazdnujeme tak objekty z $productsWillGoToProductsToOrder
						for ($i = 0; $i < $q; $i++) {
							$pr = $productsWillGoToProductsToOrder[array_key_first($productsWillGoToProductsToOrder)];
							if ($pr !== null) {
								$pr->order = $p->order;
								$this->em->persist($pr);
								$flush[] = $pr;
								unset($productsWillGoToProductsToOrder[array_key_first($productsWillGoToProductsToOrder)]);
								$k++;
							}
						}

						$productsCountWillGoToProductsToOrder -= $q; // odkrojime tu čast produktů, co již byla přiřazena
					}
				}
			}

			$this->em->persist($productToOrder);
			$flush[] = $productToOrder;
		}

		// ulozit jen ty produkty order item, ktere jsou fyzicky na sklade a nejdou do produktů k objednání
		foreach ($productsWillGoToStock as $supplyProduct) {
			$supplyProduct->order = null;
			$this->em->persist($supplyProduct);
			$flush[] = $supplyProduct;
		}

		// testuji, zda se nejedna o zbozi, ktere bylo nakoupeno jako nedostupne, pak totiz nejde ani do skladu ani do zbozi k objednani k jine objednavce (jelikoz neni fyzicky kus)
		if ($productsCountWillGoToStock === 0 && $b === 0) {
			$productsCountWillGoToStock = $quantity;
		}
		// upraveni poctu ks u produktu = (to co jde do skladu pro budouci objednavky + chybejici kusy tzn. co nesly ani do skladu ani do produktu k objednani)
		//		$product->quantity += $productsCountWillGoToStock + ($quantity - $k - $productsCountWillGoToStock);
		$product->quantity += $productsCountWillGoToStock + ($quantity - $productsCountWillGoToStock);
		$this->em->persist($product);
		$flush[] = $product;

		$this->em->flush();
	}

}
