<?php declare(strict_types = 1);

namespace EshopStock\AdminModule\Components;

use Core\Model\Http\CsvResponse;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use EshopCatalog\AdminModule\Model\Manufacturers;
use EshopCatalog\AdminModule\Model\Suppliers;
use EshopCatalog\Model\Entities\Manufacturer;
use EshopCatalog\Model\Entities\Supplier;
use EshopStock\Model\Entities\Stock;
use EshopStock\Model\Repository\Stocks;
use EshopStock\Model\Repository\Supplies;
use Nette\Utils\ArrayHash;
use Nette\Utils\DateTime;

class ToDateForm extends BaseControl
{
	protected Stocks        $stocksService;
	protected Stock         $stock;
	protected Manufacturers $manufacturers;
	protected Suppliers     $suppliers;
	protected Supplies      $supplies;
	protected array         $drawableConfig = [
		'stockValue'      => false,
		'itemsSimplified' => false,
		'itemsDetail'     => false,
	];

	public function __construct(
		int           $stockId,
		Stocks        $stocksService,
		Manufacturers $manufacturers,
		Suppliers     $suppliers,
		Supplies      $supplies
	)
	{
		$this->stocksService = $stocksService;
		$this->stock         = $stocksService->get($stockId);
		$this->manufacturers = $manufacturers;
		$this->suppliers     = $suppliers;
		$this->supplies      = $supplies;
		$this->supplies->createQueryFilter($this->stock->getId())->getSimplifiedResult();
	}

	public function render(): void
	{
		$this->template->drawableConfig = $this->drawableConfig;
		$this->template->render($this->getTemplateFile());
	}

	public function handleDownloadStockValueDetailCsv(string $payload): void
	{
		$values = unserialize(base64_decode($payload));

		$filter = $this->supplies->createQueryFilter($this->stock->getId());
		$filter->toDate($values->toDate);

		if ($values->sortBy === 'code') {
			$filter->addOrderByCode();
		} else {
			$filter->addOrderByDateSupply('desc');
		}

		if ($values->manufacturer) {
			$filter->withManufacturer($values->manufacturer);
		}

		if ($values->supplier) {
			$filter->withSupplier($values->supplier);
		}

		$csv = '';
		foreach ($filter->getSum() as $k => $v) {
			if (empty($v['manufacturer'])) {
				continue;
			}
			$csv .= '"' . $v['manufacturer'] . '";"' . $v['purchasePrice'] . '";"' . $v['count'] . '"' . PHP_EOL;
		}

		$date = date('Y-m-d');
		CsvResponse::sendResponse("stock-value-by-manufacturer-{$date}.csv", $csv);
	}

	public function createComponentFilterForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAjax();

		$form->addDatePicker('toDate', 'eshopStock.toDateForm.toDate')
			->setDefaultValue(new DateTime)
			->setRequired();
		$form->addSelect('manufacturer', 'eshopStock.toDateForm.manufacturer.caption')->setPrompt('eshopStock.toDateForm.manufacturer.prompt');
		$form->addSelect('supplier', 'eshopStock.toDateForm.supplier.caption')->setPrompt('eshopStock.toDateForm.supplier.prompt');
		$form->addSelect('sortBy', 'eshopStock.toDateForm.sortBy.caption', $this->getSortByPairs());

		$form->addSubmit('stockValue', 'eshopStock.toDateForm.submits.stockValue');
		$form->addSubmit('stockValueDetail', 'eshopStock.toDateForm.submits.stockValueDetail');
		$form->addSubmit('stockValueDetailCsv');
		$form->addSubmit('itemsSimplified', 'eshopStock.toDateForm.submits.itemsSimplified');
		$form->addSubmit('itemsDetail', 'eshopStock.toDateForm.submits.itemsDetail');

		$form->getComponent('manufacturer')->setItems([null => null] + $this->getManufacturersPairs());
		$form->getComponent('supplier')->setItems([null => null] + $this->getSuppliersPairs());

		$form->onSuccess[] = [$this, 'onSuccess'];

		return $form;
	}

	public function onSuccess(BaseForm $form, ArrayHash $values): void
	{
		$filter = $this->supplies->createQueryFilter($this->stock->getId());
		$filter->toDate($values->toDate);

		if ($values->sortBy === 'code') {
			$filter->addOrderByCode();
		} else {
			$filter->addOrderByDateSupply('desc');
		}

		if ($values->manufacturer) {
			$filter->withManufacturer($values->manufacturer);
		}

		if ($values->supplier) {
			$filter->withSupplier($values->supplier);
		}

		if ($form->getComponent('stockValue')->isSubmittedBy()) {
			$result                             = $filter->getSumSimplified();
			$this->drawableConfig['stockValue'] = true;
			$this->template->itemsCount         = $result['itemsCount'];
			$this->template->itemsValue         = $result['itemsValue'];
		} else if ($form->getComponent('itemsSimplified')->isSubmittedBy()) {
			$this->drawableConfig['itemsSimplified'] = true;
			$this->template->items                   = $filter->getSimplifiedResult();
		} else if ($form->getComponent('stockValueDetailCsv')->isSubmittedBy()) {
			$this->redirect('downloadStockValueDetailCsv', base64_encode(serialize($values)));
		} else if ($form->getComponent('itemsDetail')->isSubmittedBy()) {
			$this->drawableConfig['itemsDetail'] = true;
			$this->template->items               = $filter->getFullResult();
		} else if ($form->getComponent('stockValueDetail')->isSubmittedBy()) {
			$this->drawableConfig['stockValueDetail'] = true;
			$this->template->items                    = $filter->getSum();
		}

		$this->redrawControl('resultSnippet');
	}

	/**
	 * @return Supplier[]
	 */
	protected function getSuppliersAssoc(): array
	{
		$result = [];
		foreach ($this->suppliers->getEr()->createQueryBuilder('s')
			         ->orderBy('s.name')->getQuery()->getResult() as $supplier) {
			if ((bool) $supplier->isPublished) {
				$result[$supplier->getId()] = $supplier;
			}
		}

		return $result;
	}

	/**
	 * @return string[]
	 */
	protected function getSuppliersPairs(): array
	{
		$result = [];
		foreach ($this->getSuppliersAssoc() as $key => $supplier) {
			$result[$key] = $supplier->name;
		}

		return $result;
	}

	/**
	 * @return Manufacturer[]
	 */
	protected function getManufacturersAssoc(): array
	{
		$result = [];
		foreach ($this->manufacturers->getEr()->createQueryBuilder('m')
			         ->orderBy('m.name')->getQuery()->getResult() as $manufacturer) {
			if ((bool) $manufacturer->isPublished) {
				$result[$manufacturer->getId()] = $manufacturer;
			}
		}

		return $result;
	}

	/**
	 * @return string[]
	 */
	protected function getManufacturersPairs(): array
	{
		$result = [];
		foreach ($this->getManufacturersAssoc() as $key => $manufacturer) {
			$result[$key] = $manufacturer->name;
		}

		return $result;
	}

	/**
	 * @return array<string, string>
	 */
	protected function getSortByPairs(): array
	{
		return [
			'date' => 'eshopStock.toDateForm.sortBy.items.date',
			'code' => 'eshopStock.toDateForm.sortBy.items.code',
		];
	}

}
