<?php declare(strict_types=1);

namespace EshopStock\Model\Subscribers;

use Core\Model\Event\FormSuccessEvent;
use Core\Model\UI\AbstractPresenter;
use EshopOrders\AdminModule\Components\Order\OrdersGrid;
use EshopOrders\AdminModule\Components\Order\OrderStatusesGrid;
use EshopOrders\AdminModule\Components\Order\OrderStatusForm;
use EshopOrders\AdminModule\Model\Event\OrderStatusEvent;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderStatus;
use EshopStock\Model\Repository\SupplyProducts;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

class OrderStatusSubscriber implements EventSubscriberInterface
{
	protected SupplyProducts $supplyProducts;

	public function __construct(SupplyProducts $supplyProducts)
	{
		$this->supplyProducts = $supplyProducts;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			OrderStatusForm::class . '::formSuccess' => 'formSuccess',
			OrderStatusesGrid::class . '::beforeDelete' => 'beforeDelete',
			OrdersGrid::class . '::changeStatus' => 'changeStatus'
		];
	}

	public function formSuccess(FormSuccessEvent $event): void
	{
		/** @var OrderStatus $entity */
		$entity = $event->custom['entity'];
		$order = $entity->getOrder();
		$presenter = $event->presenter;
		/** @var string $value */
		$value = $event->values->status;

		$this->onOrderSetCanceled($order, $value, $presenter);
	}

	public function beforeDelete(OrderStatusEvent $event): void
	{
		$isOk = $this->onDeleteStatus($event->orderStatus);

		if ($isOk && $event->presenter) {
			$event->presenter->flashMessageSuccess('eshopStock.messages.addedToOrder');
		}
	}

	public function onDeleteStatus(OrderStatus $orderStatus): bool
	{
		if ($orderStatus->getStatus()->getId() !== OrderStatus::STATUS_CANCELED) {
			return false;
		}

		// pocet statusů canceled
		$canceledStatusCount = 0;
		foreach ($orderStatus->getOrder()->getOrderStatuses()->toArray() as $os) {
			if ($os->getStatus()->getId() === OrderStatus::STATUS_CANCELED && !$os->isDeleted()) {
				$canceledStatusCount++;
			}
		}

		// produkty do objednavky se vraceji az pri poslednim ruseni storna objednavky
		if ($canceledStatusCount === 1) {
			$this->supplyProducts->assignOrder($orderStatus->getOrder());
			return true;
		}

		return false;
	}

	public function changeStatus(\EshopOrders\Model\Event\OrderStatusEvent $event): void
	{
		$this->onOrderSetCanceled($event->order, $event->status, $event->presenter);
	}

	public function onOrderSetCanceled(Order $order, string $value, ?AbstractPresenter $presenter): void
	{
		$isCanceled = false;
		/** @var OrderStatus $orderStatus */
		foreach ($order->getOrderStatuses()->toArray() as $orderStatus) {
			if ($orderStatus->getStatus()->getId() === OrderStatus::STATUS_CANCELED && !$orderStatus->isDeleted()) {
				$isCanceled = true;
			}
		}

		if ($value === OrderStatus::STATUS_CANCELED && !$isCanceled) {
			$this->supplyProducts->removeOrder($order);
			if ($presenter) {
				$presenter->flashMessageSuccess('eshopStock.messages.addedToStock');
			}
		}
	}

}