<?php declare(strict_types = 1);

use Contributte\Translation\Translator;
use Core\AdminModule\Model\Sites;
use Core\Model\BootstrapTest;
use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Helpers\Strings;
use Doctrine\ORM\Query\Expr\Join;
use EshopCatalog\AdminModule\Model\AvailabilityService;
use EshopCatalog\AdminModule\Model\Categories;
use EshopCatalog\AdminModule\Model\Suppliers;
use EshopCatalog\Model\Entities\Product;
use EshopCatalog\Model\Entities\ProductInSite;
use EshopCatalog\Model\Entities\ProductTexts;
use EshopCatalog\Model\Entities\VatRate;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderItem;
use EshopOrders\Model\Entities\OrderStatus;
use EshopOrders\Model\Entities\Status;
use EshopOrders\Model\OrderItems;
use EshopOrders\Model\Orders;
use EshopStock\Model\Entities\ProductsToOrderOrders;
use EshopStock\Model\Entities\ProductToOrder;
use EshopStock\Model\Entities\SupplyProduct;
use EshopStock\Model\Repository\Stocks;
use EshopStock\Model\Repository\Supplies;
use EshopStock\Model\Subscribers\OrderStatusSubscriber;
use Nette\DI\Container;
use Nette\Utils\Arrays;
use Tester\Assert;
use Tester\TestCase;
use EshopOrders\ApiModule\Api\V1\Model\Orders as ApiOrders;
use Users\Model\Entities\User;

require __DIR__ . '/../../../../autoload.php';

/**
 * @testCase
 */
class OrderInteractionWithStockTest extends TestCase
{
	/**
	 * TODO darky??
	 */
	protected ?Product $product = null;
	protected EntityManagerDecorator $em;
	protected Categories $categories;
	protected Sites $sites;
	protected Translator $translator;
	protected Supplies $supplies;
	protected Suppliers $suppliers;
	protected Stocks $stocks;
	protected AvailabilityService $availabilityService;
	protected Container $container;
	protected Orders $orders;
	protected \EshopOrders\AdminModule\Model\Orders $adminOrders;
	protected OrderItems $orderItems;
	protected OrderStatusSubscriber $orderStatusSubscriber;

	public function __construct(Container $container)
	{
		$this->container = $container;
		$this->em = $container->getByType(EntityManagerDecorator::class);
		$this->categories = $container->getByType(Categories::class);
		$this->sites = $container->getByType(Sites::class);
		$this->translator = $container->getByType(Translator::class);
		$this->supplies = $container->getByType(Supplies::class);
		$this->suppliers = $container->getByType(Suppliers::class);
		$this->stocks = $container->getByType(Stocks::class);
		$this->availabilityService = $container->getByType(AvailabilityService::class);
		$this->orders = $container->getByType(Orders::class);
		$this->orderItems = $container->getByType(OrderItems::class);
		$this->orderStatusSubscriber = $container->getByType(OrderStatusSubscriber::class);
		$this->adminOrders = $container->getByType(\EshopOrders\AdminModule\Model\Orders::class);
	}

	protected function createProduct(): void
	{
		$availability = 'soldOut';

		$this->em->beginTransaction();

		try {
			$vatRate = new VatRate;
			$vatRate->rate = 21;
			$vatRate->name = 'TAX';
			$this->em->persist($vatRate);

			$site = $this->sites->getEr()->createQueryBuilder('s')
								->join('s.texts', 'st', Join::WITH, 'st.lang = :lang')
								->where("s.isVisible = 1 AND st.isActive = 1 AND st.isDefault = 1 AND st.defaultCountry = 'cz' AND st.defaultCurrency = 'czk'")
								->setParameter('lang', $this->translator->getLocale())
								->setMaxResults(1)
								->getQuery()->getOneOrNullResult();

			$category = $this->categories->getEr()->createQueryBuilder('c')
										 ->orderBy('c.id', 'desc')
										 ->setMaxResults(1)
										 ->getQuery()->getOneOrNullResult();

			if ($category === null) {
				throw new RuntimeException('Category required');
			}

			if ($site === null) {
				throw new RuntimeException('Site required');
			}

			$this->product = new Product;
			$this->product->isPublished = 1;
			$this->product->quantity = 0;
			$this->product->unlimitedQuantity = true;
			$this->product->price = 200;
			$this->product->retailPrice = 300;
			$this->product->purchasePrice = 100;

			$ps = new ProductInSite($this->product, $site);
			$ps->category = $category;
			$this->product->sites->add($ps);
			$this->em->persist($ps);

			$av = $this->availabilityService->getEr()->findOneBy(['ident' => $availability]);
			$this->product->setVateRate($vatRate);
			$this->product->setAvailability($av);
			$this->product->availabilityAfterSoldOut = $av;

			$productTexts = [];
			foreach ($this->translator->getLocalesWhitelist() as $locale) {
				$pt = new ProductTexts($this->product, $locale);
				$pt->setName('Nový produkt');
				$productTexts[] = $pt;
			}
			$this->product->setProductTexts($productTexts);

			$this->em->persist($this->product)->flush();

			$this->em->commit();

		} catch (Exception $exception) {
			$this->em->rollback();
			throw new $exception;
		}
	}

	protected function createOrderItem(Order $order, int $quantity): OrderItem
	{
		$lang = $this->translator->getLocale();
		$orderItem = new OrderItem($this->product, $order);
		$orderItem->setCode1($this->product->code1);
		$orderItem->addOrderItemText($lang);
		$orderItem->getOrderItemText($lang)->setName($this->product->getTexts()->first()->name);
		$orderItem->setQuantity($quantity);
		$orderItem->setPrice($this->product->price);
		$orderItem->setVatRate($this->product->vatRate->rate);
		$this->orderItems->addOrderItem($orderItem);

		return $orderItem;
	}

	protected function createSupply(int $quantity): void
	{
		$productId = $this->product->getId();
		$purchasePrice = Strings::formatEntityDecimal(150);

		$products[$productId] = [
			'quantity'      => $quantity,
			'purchasePrice' => $purchasePrice,
			'recyclingFee'  => 1
		];

		$result = $this->supplies->createSupply($this->stocks->getDefaultStock(), $this->suppliers->get(3), 'Aa12345', new \Nette\Utils\DateTime, $products);
		Assert::true($result, 'Create supply method has error');
	}

	/**
	 * @return int<0, max>
	 */
	protected function getPhysicalStockQuantity(int $forOrderId = null): int
	{
		$qb = $this->em->getRepository(SupplyProduct::class)->createQueryBuilder('sp')
					   ->select('COUNT(sp.id)')
					   ->andWhere('sp.order IS NULL and sp.writtenOffDate IS NULL')
					   ->andWhere('sp.product = ' . $this->product->getId());

		if ($forOrderId !== null) {
			$qb->andWhere('sp.order = ' . $forOrderId);
		}

		return $qb->getQuery()->getSingleScalarResult();
	}

	protected function getProductQuantity(): int
	{
		return $this->em->getRepository(Product::class)->createQueryBuilder('p')
						->select('p.quantity')
						->where('p.id = ' . $this->product->getId())
						->getQuery()->getSingleScalarResult();
	}

	/**
	 * @return array{quantity: int, orders: int[]}
	 */
	protected function getMissingPiecesInOrders(): array
	{
		$result = ['quantity' => 0, 'orders' => []];

		/** @var ProductToOrder|null $pto */
		$pto = $this->em->getRepository(ProductToOrder::class)->findOneBy(['product' => $this->product->getId()]);

		if ($pto === null) {
			return $result;
		}

		$result['quantity'] = $pto->quantity;

		/** @var ProductsToOrderOrders $o */
		foreach ($pto->orders->toArray() as $o) {
			$id = $o->order->getId();
			$result['orders'][$id] = $id;
		}
		return $result;
	}

	protected function createAnotherOrder(): Order
	{
		$ordersService = $this->container->getByType(ApiOrders::class);
		$qb = $this->em->getRepository(Product::class)->createQueryBuilder('p');
		/** @var Product $product */
		$product = $qb->andWhere('p.isPublished = 1')->andWhere($qb->expr()->neq('p.id', $this->product->getId()))
					  ->setMaxResults(1)->getQuery()->getOneOrNullResult();

		$data = $this->getOrderData(1);
		$data['checkoutProducts'][0]['id'] = $product->getId();
		$data['checkoutProducts'][0]['name'] = $product->getTexts()->first()->name;
		$data['checkoutProducts'][0]['price'] = $product->price;
		$data['checkoutProducts'][0]['code1'] = $product->code1;
		$data['checkoutProducts'][0]['code2'] = $product->code2;
		$data['checkoutProducts'][0]['ean'] = $product->ean;
		$data['checkoutProducts'][0]['quantity'] = $product->quantity;
		$data['checkoutProducts'][0]['isPublished'] = $product->isPublished;
		$data['checkoutProducts'][0]['vatRate'] = $product->vatRate->rate;
		$data['checkoutProducts'][0]['count'] = 1;
		$data['totalPrice'] = (float) $product->price;

		return $ordersService->createForCheckout($data);
	}

	protected function getOrderData(int $quantity): array
	{
		$price = $this->product->price;
		$totalPrice = $quantity * $price;

		return [
			'checkoutProducts' => [
				0 => [
					'id'          => $this->product->getId(),
					'name'        => $this->product->getTexts()->first()->name,
					'price'       => $price,
					'code1'       => $this->product->code1,
					'code2'       => $this->product->code2,
					'ean'         => $this->product->ean,
					'quantity'    => $this->product->quantity,
					'isPublished' => $this->product->isPublished,
					'vatRate'     => $this->product->vatRate->rate,
					'count'       => $quantity,
					'sale'        => [
						'all' => [
							'name'        => '',
							'staticSale'  => 0,
							'percentSale' => 0,
							'note'        => '',
							'saleCode'    => '',
						],
						'one' => [
							'name'        => '',
							'staticSale'  => 0,
							'percentSale' => 0,
							'note'        => '',
							'saleCode'    => '',
						],
					],
				],
			],
			'globalSales'      => [],
			'payment'          => 'storeCash',
			'spedition'        => 'pickup',
			'customer'         => [
				'customerId'      => null,
				'customerCreated' => false,
				'addressDelivery' => [
					'firstName' => '',
					'lastName'  => '',
					'email'     => '',
					'phone'     => '',
					'street'    => '',
					'city'      => '',
					'postal'    => '',
					'country'   => '',
					'company'   => '',
				],
				'addressInvoice'  => [
					'firstName' => '',
					'lastName'  => '',
					'email'     => '',
					'phone'     => '',
					'street'    => '',
					'city'      => '',
					'postal'    => '',
					'country'   => '',
					'company'   => '',
					'idNumber'  => '',
					'vatNumber' => '',
				],
			],
			'totalPrice'       => $totalPrice,
			'transactionId'    => null,
		];
	}

	/**
	 * @param bool[] $values
	 */
	protected static function assertAllTrue(array $values, string $desc = null): void
	{
		Assert::true(Arrays::every($values, static fn(bool $val) => $val === true), $desc);
	}

	public function setUp(): void
	{
		$this->em->clear();
	}

	public function testInteractions(): void
	{
		$ordersService = $this->container->getByType(ApiOrders::class);
		$actualQuantity = 10;
		$missingQuantity = 0; // ve zbozi k objednani
		$missingPiecesInOrders = [];
		$this->createProduct();

		// ----------------------------- Prvni naskladneni
		$this->createSupply($actualQuantity);
		self::assertAllTrue([
			($pQuantity = $this->getProductQuantity()) === ($psq = $this->getPhysicalStockQuantity()),
			$pQuantity === $actualQuantity,
			$psq === $actualQuantity,
			($mpo = $this->getMissingPiecesInOrders())['orders'] === $missingPiecesInOrders,
			$mpo['quantity'] === $missingQuantity
		], 'Chyba po prvnim naskladneni');

		// ----------------------------- Vytvoreni prvni objednavky (c. 1) (quantity produktu v plusu)
		$purchaseQuantity = 1;
		$actualQuantity -= $purchaseQuantity;
		$firstOrder = $ordersService->createForCheckout($this->getOrderData($purchaseQuantity));
		self::assertAllTrue([
			($pQuantity = $this->getProductQuantity()) === ($psq = $this->getPhysicalStockQuantity()),
			$pQuantity === $actualQuantity,
			$psq === $actualQuantity,
			($mpo = $this->getMissingPiecesInOrders())['orders'] === $missingPiecesInOrders,
			$mpo['quantity'] === $missingQuantity
		], 'Chyba po prvni objednavce');

		// ----------------------------- Vytvoreni druhe objednavky (c. 2) (quantity produktu v minusu)
		$diffQuantity = 5;
		$purchaseQuantity = $actualQuantity + $diffQuantity;
		$physicalProductQuantityInOrderTwo = $purchaseQuantity - $diffQuantity;
		$missingProductQuantityInOrderTwo = $diffQuantity;
		$missingQuantity += -$diffQuantity;
		$actualQuantity -= $purchaseQuantity;
		$partialBalanceOrder = $ordersService->createForCheckout($this->getOrderData($purchaseQuantity));
		$missingPiecesInOrders[$partialBalanceOrder->getId()] = $partialBalanceOrder->getId();
		self::assertAllTrue([
			$this->getProductQuantity() === $actualQuantity,
			$this->getPhysicalStockQuantity() === 0,
			($mpo = $this->getMissingPiecesInOrders())['orders'] === $missingPiecesInOrders,
			$mpo['quantity'] === $missingQuantity
		], 'Chyba po druhe objednavce, kdy bylo objednano mnozstvi, jez castecne je a castecne neni na sklade');

		// ----------------------------- Pridani produktu do jine objednavky (c. 3) (quantity produktu vice v minusu)
		$purchaseQuantity = 5;
		$missingQuantity += -$purchaseQuantity;
		$actualQuantity -= $purchaseQuantity;
		$missingProductQuantityInOrderThree = $purchaseQuantity;
		$orderToAddProduct = $this->createAnotherOrder();
		$missingPiecesInOrders[$orderToAddProduct->getId()] = $orderToAddProduct->getId();
		$orderItemIdOfOrderThree = $this->createOrderItem($orderToAddProduct, $purchaseQuantity)->getId();
		self::assertAllTrue([
			$this->getProductQuantity() === $actualQuantity,
			$this->getPhysicalStockQuantity() === 0,
			($mpo = $this->getMissingPiecesInOrders())['orders'] === $missingPiecesInOrders,
			$mpo['quantity'] === $missingQuantity
		], 'Chyba po pridani produktu do objednavky c. 3, kdy bylo objednano mnozstvi, jez neni na sklade');

		// ----------------------------- Odebrani produktu z objednavky c. 2 (quantity produktu se navysi jen o trochu)
		/**
		 * Z objednavky c. 2 se fyzicky pritomne produkty presunou na vykryti objednavky c. 3
		 * a objednavka c. 2 zmizi ze zbozi k objednani a produkt quantity bude o ten zbytek castecne navysen
		 */
		/** @var OrderItem $orderItemToRemove */
		$orderItemToRemove = $this->orders->get($partialBalanceOrder->getId())->getOrderItems()->first();
		$orderItemToRemoveId = $orderItemToRemove->getId();
		$actualQuantity += $physicalProductQuantityInOrderTwo + $missingProductQuantityInOrderTwo; // pricteme celou quantitu z objednavky 2
		$productQuantityToBalanceOrderThree = ($physicalProductQuantityInOrderTwo - $missingProductQuantityInOrderThree) <= 0 // zjistime kolik produktu muze jit na vykryvani
			? $physicalProductQuantityInOrderTwo
			: $missingProductQuantityInOrderThree;

		$missingQuantity += $productQuantityToBalanceOrderThree + $missingProductQuantityInOrderTwo;
		unset($missingPiecesInOrders[$orderItemToRemove->order->getId()]); // v obj. c. 2 jiz nic nechybi, jde pryc ze zbozi k objednani
		if ($missingQuantity >= 0) {
			$missingPiecesInOrders = [];
			$missingQuantity = 0;
		}
		$this->em->clear();
		$this->orderItems->removeOrderItem($orderItemToRemoveId);
		self::assertAllTrue([
			$this->getProductQuantity() === $actualQuantity,
			$this->getPhysicalStockQuantity() === $actualQuantity,
			// fyzicky stav by mel byt nad nulou, tudiz muzeme testovat, zda odpovida quantity produktu
			($mpo = $this->getMissingPiecesInOrders())['orders'] === $missingPiecesInOrders,
			$mpo['quantity'] === $missingQuantity
		], 'Chyba po odebrani produktu z objednavky c. 2, kdy se mely fyzicke produkty pouzit na vykryti objednavky c. 3');

		// ----------------------------- Navyseni poctu ks u obj. c. 1 (quantity produktu jde do minusu)
		$this->em->clear();
		$increaseByQuantity = 10;
		$missingPiecesInOrders[$firstOrder->getId()] = $firstOrder->getId();
		$missingQuantity -= abs(max(0, $actualQuantity) - $increaseByQuantity); // pokud je neco na sklade, tak stanovi jen chybejici rozdil. Pokud nic neni nasklade, tak se vezme cele nevysovane mnozstvi
		$actualQuantity -= $increaseByQuantity;
		/** @var OrderItem $orderItemToIncrease */
		$orderItemToIncrease = $this->orders->get($firstOrder->getId())->getOrderItems()->first();
		$orderItemToIncrease->setQuantity($orderItemToIncrease->getQuantity() + $increaseByQuantity);
		$this->orderItems->saveOrderItem($orderItemToIncrease);
		self::assertAllTrue([
			$this->getProductQuantity() === $actualQuantity,
			$this->getPhysicalStockQuantity() === 0, // aktualni stav je pod nulou, tudiz fyzicky by mel byt 0
			($mpo = $this->getMissingPiecesInOrders())['orders'] === $missingPiecesInOrders,
			$mpo['quantity'] === $missingQuantity
		], 'Chyba po navyseni poctu ks objednavky c. 1, kdy bylo navyseno o takove mnozstvi, ktere nebylo aktualne na sklade');

		// ----------------------------- Naskladneni produktu (quantity produktu ze zvedne, ale porad bude pod nulou) a budou se castecne vykryvat obj. c. 1
		$this->em->clear();
		$supplyQuantity = 4;
		$actualQuantity += $supplyQuantity;
		$missingQuantity += $supplyQuantity;
		// neni treba kontrolovat zmenu v objednavkach kde chybi, v tomto pripade se nic nemeni
		$this->createSupply($supplyQuantity);
		self::assertAllTrue([
			$this->getProductQuantity() === $actualQuantity,
			$this->getPhysicalStockQuantity() === 0, // aktualni stav je pod nulou, tudiz fyzicky by mel byt 0
			($mpo = $this->getMissingPiecesInOrders())['orders'] === $missingPiecesInOrders,
			$mpo['quantity'] === $missingQuantity
		], 'Chyba po naskladneni produktu. Naskladnovane mnozstvi melo jit na castecne vykryti objednavky c. 1');

		// ----------------------------- Snizeni poctu ks u obj. c. 3, kdy fyzicke kusy jdou na castecne vykryti obj. c. 1
		$this->em->clear();
		$diffQuantity = 1;
		$actualQuantity += $diffQuantity;
		$missingQuantity += $diffQuantity;
		// neni treba kontrolovat zmenu v objednavkach kde chybi, v tomto pripade se nic nemeni
		$orderItemToDecrease = $this->orderItems->get($orderItemIdOfOrderThree);
		$orderItemToDecrease->setQuantity($orderItemToDecrease->getQuantity() - $diffQuantity);
		$this->orderItems->saveOrderItem($orderItemToDecrease);
		self::assertAllTrue([
			$this->getProductQuantity() === $actualQuantity,
			$this->getPhysicalStockQuantity() === 0, // aktualni stav je pod nulou, tudiz fyzicky by mel byt 0
			($mpo = $this->getMissingPiecesInOrders())['orders'] === $missingPiecesInOrders,
			$mpo['quantity'] === $missingQuantity
		], 'Chyba po snizeni poctu ks u obj. c. 3, kdy melo jit snizene mnozstvi na vykryti obj. c. 1');

		// ----------------------------- Storno obj. c. 3, kdy fyzicke kusy jdou na uplne vykryti obj. c. 1
		$this->em->clear();
		$orderThree = $this->orders->get($orderToAddProduct->getId());
		$canceledQuantity = $orderThree->getOrderItemByProductId($this->product->getId())->getQuantity();
		$actualQuantity += $canceledQuantity;
		$missingQuantity += $canceledQuantity;
		unset($missingPiecesInOrders[$orderToAddProduct->getId()]);
		if ($missingQuantity >= 0) {
			$missingPiecesInOrders = [];
			$missingQuantity = 0;
		}
		$this->orderStatusSubscriber->onOrderSetCanceled($orderThree, OrderStatus::STATUS_CANCELED, null);
		$status = $this->em->getRepository(Status::class)->getReference(OrderStatus::STATUS_CANCELED);
		$userRef = $this->em->getRepository(User::class)->getReference(1);
		$orderStatusCanceled = new OrderStatus($orderThree, $status, $userRef);
		$this->em->persist($orderStatusCanceled)->flush();
		self::assertAllTrue([
			($pQuantity = $this->getProductQuantity()) === ($psq = $this->getPhysicalStockQuantity()),
			// fyzicka quantita je nad nulou, mzeme porovnavat product quantity s fyzickou
			$pQuantity === $actualQuantity,
			$psq === $actualQuantity,
			($mpo = $this->getMissingPiecesInOrders())['orders'] === $missingPiecesInOrders,
			$mpo['quantity'] === $missingQuantity
		], 'Chyba po stornovani obj. c. 3, kdy se fyzicke produkty mely presunout do obj. c. 1');

		// ----------------------------- Vytvoreni obj. c. 4 (quantity produktu je minusu) a zruseni prechoziho storna obj. c. 3
		$this->em->clear();
		$diffQuantity = 2;
		$purchaseQuantity = $actualQuantity + $diffQuantity;
		$actualQuantity -= $purchaseQuantity;
		$missingQuantity -= $diffQuantity;
		$orderFour = $ordersService->createForCheckout($this->getOrderData($purchaseQuantity));
		$missingPiecesInOrders[$orderFour->getId()] = $orderFour->getId();
		self::assertAllTrue([
			$this->getProductQuantity() === $actualQuantity,
			$this->getPhysicalStockQuantity() === 0, // aktualni stav je pod nulou, tudiz fyzicky by mel byt 0
			($mpo = $this->getMissingPiecesInOrders())['orders'] === $missingPiecesInOrders,
			$mpo['quantity'] === $missingQuantity
		], 'Chyba po vytvoreni obj. c. 4, kdy se objednava mnozstvi, jez je na sklade jen z casti');
		// zruseni storna obj. c. 3, kdy objednani mnozstvi neni na sklade, tudiz je potreba cele mnozstvi zapsat do zbozi k objednani
		$this->em->clear();
		$canceledQuantity = $orderThree->getOrderItemByProductId($this->product->getId())->getQuantity();
		$physicalProductQuantityInOrderThree = $this->getPhysicalStockQuantity($orderThree->getId());
		$missingProductQuantityInOrderThree = $canceledQuantity - $physicalProductQuantityInOrderThree;
		if ($this->orderStatusSubscriber->onDeleteStatus($this->em->getRepository(OrderStatus::class)
																  ->find($orderStatusCanceled->getId()))) {
			$this->em->flush();
			$actualQuantity -= $canceledQuantity;
			$missingQuantity -= $missingProductQuantityInOrderThree;
			$missingPiecesInOrders[$orderThree->getId()] = $orderThree->getId();
			self::assertAllTrue([
				$this->getProductQuantity() === $actualQuantity,
				$this->getPhysicalStockQuantity() === 0, // aktualni stav je pod nulou, tudiz fyzicky by mel byt 0
				($mpo = $this->getMissingPiecesInOrders())['orders'] === $missingPiecesInOrders,
				$mpo['quantity'] === $missingQuantity
			], 'Chyba po zruseni storna obj. c. 3, produkty pro objednavku c. 3 nejsou k dispozici a proto vse konci ve zbozi k objednani');
		}

		// ----------------------------- Vytvoreni ODD z obj. c. 4, nejprve se ale naskladni abychom meli produkt v plusu
		$this->em->clear();
		// naskladnime aby byl produkt quantity v plusu
		$supplyQuantity = abs($missingQuantity) + 1;
		$actualQuantity += $supplyQuantity;
		$missingQuantity += $supplyQuantity;
		if ($missingQuantity >= 0) {
			$missingPiecesInOrders = [];
			$missingQuantity = 0;
		}
		$this->createSupply($supplyQuantity);
		self::assertAllTrue([
			$this->getProductQuantity() === $actualQuantity,
			$this->getPhysicalStockQuantity() >= 0,
			($mpo = $this->getMissingPiecesInOrders())['orders'] === $missingPiecesInOrders,
			$mpo['quantity'] === $missingQuantity
		], 'Chyba po naskladeni pred vytvorenim odd');
		// tvorba odd
		$this->em->clear();
		$orderFour = $this->orders->get($orderFour->getId());
		$ctdOrderItem = $orderFour->getOrderItemByProductId($this->product->getId());
		$ctdQuantity = $ctdOrderItem->getQuantity();
		$actualQuantity += $ctdQuantity;
		// neni treba kontrolovat zmenu v objednavkach kde chybi, v tomto pripade se nic nemeni
		$products = [];
		/** @var OrderItem $oi */
		foreach ($orderFour->getOrderItems() as $oi) {
			$products[$oi->getId()] = [
				'price'    => $oi->getPrice(),
				'quantity' => $oi->getQuantity(),
				'oiId'     => $oi->getId(),
			];
		}
		$this->adminOrders->createCorrectiveTaxDocument($orderFour, $products, [], $orderFour->getPayment()
																							 ->getPrice(), $orderFour->getSpedition()
																													 ->getPrice());

		self::assertAllTrue([
			($pQuantity = $this->getProductQuantity()) === ($psq = $this->getPhysicalStockQuantity()),
			// fyzicka quantita je nad nulou, mzeme porovnavat product quantity s fyzickou
			$pQuantity === $actualQuantity,
			$psq === $actualQuantity,
			($mpo = $this->getMissingPiecesInOrders())['orders'] === $missingPiecesInOrders,
			$mpo['quantity'] === $missingQuantity
		], 'Chyba po vytvoreni ODD z obj. c. 4');
	}

	public function getBalanceOrdersArg(): array
	{
		return [
			[
				[
					'firstOrderPurchaseQuantity'  => 8,
					'secondOrderPurchaseQuantity' => 5,
					'supplyQuantity'              => 2,
					'increaseByQuantity'          => 1,
					'decreaseByQuantity'          => 2,
				]
			],
			[
				[
					'firstOrderPurchaseQuantity'  => 12,
					'secondOrderPurchaseQuantity' => 6,
					'supplyQuantity'              => 2,
					'increaseByQuantity'          => 1,
					'decreaseByQuantity'          => 1,
				]
			],
			[
				[
					'firstOrderPurchaseQuantity'  => 9,
					'secondOrderPurchaseQuantity' => 4,
					'supplyQuantity'              => 2,
					'increaseByQuantity'          => 1,
					'decreaseByQuantity'          => 1,
				]
			]
		];
	}

	/**
	 * @dataProvider getBalanceOrdersArg
	 * @var array{
	 *     firstOrderPurchaseQuantity: int<1, max>,
	 *     secondOrderPurchaseQuantity: int<1, max>,
	 *     supplyQuantity: int<1, max>,
	 *     increaseByQuantity: int<1, max>,
	 *     decreaseByQuantity: int<1, max>,
	 * } $arg
	 */
	public function testBalanceOrders(array $arg): void
	{
		$ordersService = $this->container->getByType(ApiOrders::class);
		$actualQuantity = 10;
		$missingQuantity = 0; // ve zbozi k objednani
		$missingPiecesInOrders = [];
		$this->createProduct();
		$this->createSupply($actualQuantity);

		// obj 1
		$purchaseQuantity = $arg['firstOrderPurchaseQuantity'];
		$firstOrder = $ordersService->createForCheckout($this->getOrderData($purchaseQuantity));
		$missingPiecesInFirstOrder = 0;
		if ($purchaseQuantity > $actualQuantity) {
			$missingPiecesInFirstOrder = $actualQuantity <= 0 ? $purchaseQuantity : abs($purchaseQuantity - $actualQuantity);
			$missingQuantity -= $missingPiecesInFirstOrder;
			$missingPiecesInOrders[$firstOrder->getId()] = $firstOrder->getId();
		}
		$actualQuantity -= $purchaseQuantity;
		self::assertAllTrue([
			$this->getProductQuantity() === $actualQuantity,
			$this->getPhysicalStockQuantity() === max($actualQuantity, 0),
			($mpo = $this->getMissingPiecesInOrders())['orders'] === $missingPiecesInOrders,
			$mpo['quantity'] === $missingQuantity
		], 'Chyba po vytvoreni obj 1');

		// obj 2
		$purchaseQuantity = $arg['secondOrderPurchaseQuantity'];
		$secondOrder = $ordersService->createForCheckout($this->getOrderData($purchaseQuantity));
		$missingPiecesInSecondOrder = 0;
		if ($purchaseQuantity > $actualQuantity) {
			$missingPiecesInSecondOrder = $actualQuantity <= 0 ? $purchaseQuantity : abs($purchaseQuantity - $actualQuantity);
			$missingQuantity -= $missingPiecesInSecondOrder;
			$missingPiecesInOrders[$secondOrder->getId()] = $secondOrder->getId();
		}
		$actualQuantity -= $purchaseQuantity;
		self::assertAllTrue([
			$this->getProductQuantity() === $actualQuantity,
			$this->getPhysicalStockQuantity() === max($actualQuantity, 0),
			($mpo = $this->getMissingPiecesInOrders())['orders'] === $missingPiecesInOrders,
			$mpo['quantity'] === $missingQuantity
		], 'Chyba po vytvoreni obj 2');

		// naskladneni
		$supplyQuantity = $arg['supplyQuantity'];
		$this->createSupply($supplyQuantity);
		$missingQuantity += $missingQuantity >= 0 ? 0 : $supplyQuantity;
		$actualQuantity += $supplyQuantity;
		if ($missingPiecesInFirstOrder > 0) {
			$missingPiecesInFirstOrder -= min($supplyQuantity, $missingPiecesInFirstOrder);
			$supplyQuantity -= $missingPiecesInFirstOrder;
			if ($missingPiecesInFirstOrder === 0) {
				unset($missingPiecesInOrders[$firstOrder->getId()]);
			}
		}
		if ($supplyQuantity > 0 && $missingPiecesInSecondOrder > 0) {
			$missingPiecesInSecondOrder -= min($supplyQuantity, $missingPiecesInSecondOrder);
			$supplyQuantity -= $missingPiecesInSecondOrder;
			if ($missingPiecesInSecondOrder === 0) {
				unset($missingPiecesInOrders[$secondOrder->getId()]);
			}
		}
		self::assertAllTrue([
			$this->getProductQuantity() === $actualQuantity,
			$this->getPhysicalStockQuantity() === max($actualQuantity, 0),
			($mpo = $this->getMissingPiecesInOrders())['orders'] === $missingPiecesInOrders,
			$mpo['quantity'] === $missingQuantity
		], 'Chyba po naskladneni');

		// zvyseni poctu ks
		$this->em->clear();
		$increaseByQuantity = $arg['increaseByQuantity'];
		/** @var OrderItem $orderItemToIncrease */
		$orderItemToIncrease = $this->orders->get($firstOrder->getId())->getOrderItems()->first();
		$orderItemToIncrease->setQuantity($orderItemToIncrease->getQuantity() + $increaseByQuantity);
		$this->orderItems->saveOrderItem($orderItemToIncrease);
		if ($increaseByQuantity > $actualQuantity) {
			$missingPiecesInFirstOrderForThisOperation = $actualQuantity <= 0 ? $increaseByQuantity : abs($increaseByQuantity - $actualQuantity);
			$missingPiecesInFirstOrder += $missingPiecesInFirstOrderForThisOperation;
			$missingQuantity -= $missingPiecesInFirstOrderForThisOperation;
			$missingPiecesInOrders[$firstOrder->getId()] = $firstOrder->getId();
		}
		$actualQuantity -= $increaseByQuantity;
		self::assertAllTrue([
			$this->getProductQuantity() === $actualQuantity,
			$this->getPhysicalStockQuantity() === max($actualQuantity, 0),
			($mpo = $this->getMissingPiecesInOrders())['orders'] === $missingPiecesInOrders,
			$mpo['quantity'] === $missingQuantity
		], 'Chyba po zvyseni poctu ks obj 1');

		// snizeni poctu ks
		$this->em->clear();
		/** @var OrderItem $orderItemToDecrease */
		$orderItemToDecrease = $this->orders->get($secondOrder->getId())->getOrderItems()->first();
		$quantitySecondOrder = $orderItemToDecrease->getQuantity();
		$diffQuantity = $arg['decreaseByQuantity'];
		if (($quantitySecondOrder - $diffQuantity) <= 0) {
			$diffQuantity = 1;
		}
		$orderItemToDecrease->setQuantity($quantitySecondOrder - $diffQuantity);
		$this->orderItems->saveOrderItem($orderItemToDecrease);
		$actualQuantity += $diffQuantity;
		$missingPiecesInSecondOrderForThisOperation = min($missingPiecesInSecondOrder, $diffQuantity);
		$physicalPiecesInSecondOrderForThisOperation = min($quantitySecondOrder - $missingPiecesInSecondOrder, $diffQuantity - $missingPiecesInSecondOrderForThisOperation);
		$missingPiecesInSecondOrder -= $missingPiecesInSecondOrderForThisOperation;
		$missingPiecesInFirstOrder -= $physicalPiecesInSecondOrderForThisOperation;
		$missingQuantity += $physicalPiecesInSecondOrderForThisOperation + $missingPiecesInSecondOrderForThisOperation;
		if ($missingPiecesInSecondOrder <= 0) {
			unset($missingPiecesInOrders[$secondOrder->getId()]);
		}
		if ($missingPiecesInFirstOrder <= 0) {
			unset($missingPiecesInOrders[$firstOrder->getId()]);
		}
		self::assertAllTrue([
			$this->getProductQuantity() === $actualQuantity,
			$this->getPhysicalStockQuantity() === max($actualQuantity, 0),
			($mpo = $this->getMissingPiecesInOrders())['orders'] === $missingPiecesInOrders,
			$mpo['quantity'] === $missingQuantity
		], 'Chyba po snizeni poctu ks u obj 2');
	}

	public function testPositiveQuantity(): void
	{
		$ordersService = $this->container->getByType(ApiOrders::class);
		$actualQuantity = 10;
		$this->createProduct();
		$this->createSupply($actualQuantity);

		// obj 1
		$purchaseQuantity = 10;
		$firstOrder = $ordersService->createForCheckout($this->getOrderData($purchaseQuantity));
		$actualQuantity -= $purchaseQuantity;
		self::assertAllTrue([
			$this->getProductQuantity() === $actualQuantity,
			$this->getPhysicalStockQuantity() === max($actualQuantity, 0),
			($mpo = $this->getMissingPiecesInOrders())['orders'] === [],
			$mpo['quantity'] === 0
		], 'Chyba po vytvoreni objednavky 1');

		// storno
		$actualQuantity += $purchaseQuantity;
		$this->orderStatusSubscriber->onOrderSetCanceled($firstOrder, OrderStatus::STATUS_CANCELED, null);
		self::assertAllTrue([
			$this->getProductQuantity() === $actualQuantity,
			$this->getPhysicalStockQuantity() === max($actualQuantity, 0),
			($mpo = $this->getMissingPiecesInOrders())['orders'] === [],
			$mpo['quantity'] === 0
		], 'Chyba po stornovani objednavky 1');

		$status = $this->em->getRepository(Status::class)->getReference(OrderStatus::STATUS_CANCELED);
		$userRef = $this->em->getRepository(User::class)->getReference(1);
		$orderStatusCanceled = new OrderStatus($firstOrder, $status, $userRef);
		$this->em->persist($orderStatusCanceled)->flush();
		$this->em->clear();
		if ($this->orderStatusSubscriber->onDeleteStatus($this->em->getRepository(OrderStatus::class)
																  ->find($orderStatusCanceled->getId()))) {
			$this->em->flush();
			$actualQuantity -= $purchaseQuantity;
			self::assertAllTrue([
				$this->getProductQuantity() === $actualQuantity,
				$this->getPhysicalStockQuantity() === max($actualQuantity, 0),
				($mpo = $this->getMissingPiecesInOrders())['orders'] === [],
				$mpo['quantity'] === 0
			], 'Chyba po zruseni storna objednavky 1');
		}
	}

}

$container = BootstrapTest::bootForTests()->createContainer();
(new OrderInteractionWithStockTest($container))->run();