<?php declare(strict_types = 1);

namespace EshopStock\AdminModule\Components;

use Core\Model\Helpers\Strings;
use Core\Model\UI\AdminPresenter;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use DateTime;
use EshopCatalog\AdminModule\Model\Products;
use EshopCatalog\AdminModule\Model\Suppliers;
use EshopCatalog\Model\Entities\Product;
use EshopCatalog\Model\Entities\Supplier;
use EshopStock\Model\Entities\Stock;
use EshopStock\Model\Repository\ProductsToOrder;
use EshopStock\Model\Repository\Stocks;
use EshopStock\Model\Repository\Supplies;
use EshopStock\Model\Repository\TemporarySupplyProducts;
use Nette\Application\LinkGenerator;
use Nette\Http\Request;
use Nette\Utils\ArrayHash;
use Nette\Utils\Validators;

class SupplyForm extends BaseControl
{
	protected float $purchasePriceSum = 0;
	protected Stock $stock;

	public function __construct(
		int                               $stockId,
		protected Suppliers               $suppliers,
		protected Products                $products,
		protected Stocks                  $stocksService,
		protected TemporarySupplyProducts $temporarySupplyProductsService,
		protected ProductsToOrder         $productsToOrder,
		protected Supplies                $supplies,
		protected LinkGenerator           $linkGenerator,
		protected Request                 $request
	)
	{
		$this->stock = $stocksService->get($stockId);
	}

	public function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAjax();

		$suppliers = $this->getSuppliersPairs();
		uasort($suppliers, static fn($a, $b): int => $a <=> $b);

		$form->addSelect('supplier', 'eshopStock.supplyForm.supplier.caption', $suppliers)
			->setPrompt('eshopStock.supplyForm.supplier.prompt')
			->setRequired();
		$form->addText('invoiceNumber', 'eshopStock.supplyForm.invoiceNumber')
			->setRequired();
		$form->addDatePicker('dateSupply', 'eshopStock.supplyForm.dateSupply')
			->setDefaultValue(new DateTime)
			->setRequired();
		$form->addText('currencyRate', 'eshopStock.supplyForm.currencyRate')
			->setDescription('eshopStock.supplyForm.currencyRateDescription')
			->setDefaultValue(1)
			->setHtmlAttribute('min', 0)
			->setHtmlAttribute('step', 0.001)
			->addRule($form::FLOAT);
		$form->addText('item', 'eshopStock.supplyForm.addItem')
			->setHtmlAttribute('data-autocomplete-name', 'eshopStockSupplyFormItem')
			->setHtmlAttribute('data-autocomplete-keys', 'id,code1,code2,ean,name')
			->setHtmlAttribute('data-autocomplete-url', $this->linkGenerator->link('EshopCatalog:Cron:Products:loadAll'));

		$form->monitor(AdminPresenter::class, function() use ($form): void {
			$form->getComponent('item')->setHtmlAttribute('data-save-and-get-products', $this->link('saveAndGetProducts!'));
		});

		$form->addSubmit('submit', 'default.save');

		$form->onValidate[] = function(BaseForm $form, ArrayHash $values): void {

			$productHasAdded = false;
			foreach ((array) $form->getHttpData() as $key => $val) {
				if (\str_starts_with($key, 'product_')) {
					$productHasAdded = true;
					break;
				}
			}

			if (!$productHasAdded) {
				$form->addError('eshopStock.eshopStock.supplyForm.errors.productsRequired');
			}

			$this->redrawControl('supplyFormErrors');
		};

		$form->onSuccess[] = $this->onSuccess(...);

		return $form;
	}

	public function onSuccess(BaseForm $form, ArrayHash $values): void
	{
		$data     = (array) $form->getHttpData();
		$supplier = $this->suppliers->get((int) $data['supplier']);
		$products = [];

		$currencyRate = (float) $data['currencyRate'] ?: 1;

		foreach ($data as $key => $val) {
			if (!\str_starts_with((string) $key, 'product_')) {
				continue;
			}

			$productId    = (int) array_key_last(array_flip(explode('_', (string) $key)));
			$quantity     = (int) $val['quantity'];
			$recyclingFee = Strings::formatEntityDecimal($val['recyclingFee']);

			$tmp = (float) str_replace(',', '.', (string) $val['purchasePrice']);
			$tmp = round($tmp * $currencyRate, 2);

			$purchasePrice = Strings::formatEntityDecimal($tmp);

			if (array_key_exists($productId, $products)) {
				$products[$productId]['quantity']      += $quantity;
				$products[$productId]['purchasePrice'] = $purchasePrice;
				$products[$productId]['recyclingFee']  = $recyclingFee;
			} else {
				$products[$productId] = [
					'quantity'      => $quantity,
					'purchasePrice' => $purchasePrice,
					'recyclingFee'  => $recyclingFee,
				];
			}
		}

		$dateSupply = \Nette\Utils\DateTime::createFromFormat('d. m. Y', $data['dateSupply']);
		if (!$dateSupply || !$supplier) {
			return;
		}

		$this->supplies->createSupply(
			$this->stock,
			$supplier,
			$data['invoiceNumber'],
			$dateSupply,
			$products
		);
	}

	public function render(): void
	{
		$this->template->purchasePriceSum = $this->purchasePriceSum;
		$this->template->render($this->getTemplateFile());
	}

	/**
	 * @return Supplier[]
	 */
	protected function getSuppliersAssoc(): array
	{
		$result = [];
		foreach ($this->suppliers->getAll() as $supplier) {
			if ((bool) $supplier->isPublished) {
				$result[$supplier->getId()] = $supplier;
			}
		}

		return $result;
	}

	/**
	 * @return string[]
	 */
	protected function getSuppliersPairs(): array
	{
		$result = [];
		foreach ($this->getSuppliersAssoc() as $key => $supplier) {
			$result[$key] = $supplier->name;
		}

		return $result;
	}

	public function handleFindProducts(string $search): void
	{
		$term     = $search;
		$products = $this->products->getByTerm($term);
		$output   = [];
		/** @var Product $product */
		foreach ($products as $product) {
			$output[$product->getId()] = [
				'id'           => $product->getId(),
				'name'         => $product->code1 . ' | ' . $product->getText($this->translator->getLocale())->name . ' | ' . $product->getManufacturer()->name,
				'code1'        => $product->code1,
				'price'        => $product->price,
				'recyclingFee' => $product->recyclingFee,
				'vatRate'      => $product->getVateRate()->rate,
			];
		}

		$this->getPresenter()->sendJson($output);
	}

	public function handleSaveAndGetProducts(): void
	{
		$productId = (int) $this->request->getPost('productId');
		$this->temporarySupplyProductsService->addProduct($productId, 1);

		$products = $this->temporarySupplyProductsService->loadProducts();
		$form     = $this['form'];
		foreach ($products as $key => $product) {
			$container = $form->addContainer('product_' . $key . '_' . $product['productId']);
			$container->addInteger('quantity')
				->setDefaultValue($product['quantity'])
				->setRequired();
			$container->addText('purchasePrice')
				->setDefaultValue($product['purchasePrice'])
				->setRequired()
				->addRule($form::FLOAT);
			$container->addText('recyclingFee')
				->setDefaultValue($product['recyclingFee'])
				->setHtmlAttribute('tabindex', '-1')
				->addCondition($form::FILLED)
				->addRule($form::FLOAT);
		}

		$this->template->products = $products;
		$this->redrawControl('goodsListSnippet');
	}

	public function handleUpdateProduct(): void
	{
		$params       = $this->getParameters();
		$data         = [$params['name'] => $params['val']];
		$key          = (int) $params['key'];
		$currencyRate = (float) $params['currencyRate'];

		$this->temporarySupplyProductsService->updateProduct($key, $data);
		$products = $this->temporarySupplyProductsService->loadProducts();

		$purchasePriceSum = 0;
		foreach ($products as $product) {
			$purchasePrice    = floatval(str_replace(',', '.', $product['purchasePrice']));
			$quantity         = Validators::isNumeric($product['quantity']) ? ((int) $product['quantity']) : 0;
			$purchasePriceSum += $quantity * round($purchasePrice * $currencyRate, 2);
		}

		$this->purchasePriceSum = $purchasePriceSum;
		$this->redrawControl('purchasePriceSumSnippet');
	}

}
