<?php declare(strict_types = 1);

namespace EshopStock\AdminModule\Model\Subscribers;

use Core\Model\Event\Event;
use EshopOrders\AdminModule\Components\Order\OrderItemAddForm;
use EshopOrders\AdminModule\Components\Order\OrderItemForm;
use EshopOrders\AdminModule\Model\Event\OrderItemEvent;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderGift;
use EshopOrders\Model\Entities\OrderStatus;
use EshopOrders\Model\OrderItems;
use EshopStock\Model\Repository\SupplyProducts;
use Override;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

class OrderItemChangedSubscriber implements EventSubscriberInterface
{
	public function __construct(
		protected SupplyProducts $supplyProducts,
		protected OrderItems     $orderItems,
	)
	{
	}

	#[Override]
	public static function getSubscribedEvents(): array
	{
		return [
			OrderItemAddForm::class . '::formSuccess' => ['orderItemAdded', 100],
			OrderItemForm::class . '::beforeDelete'   => ['orderItemRemoved', 100],
			OrderItemForm::class . '::beforeSave'     => ['orderItemSaved', 100],
			'eshopOrders.gifts.beforeDelete'          => ['orderGiftRemoved', 100],
		];
	}

	public function orderGiftRemoved(Event $event): void
	{
		/** @var OrderGift $og */
		$og = $event->data['entity'];
		if (!$this->orderIsCanceled($og->getOrder())) {
			$this->supplyProducts->removeOrderItem($og);
		}
	}

	public function orderItemAdded(OrderItemEvent $event): void
	{
		if (!$this->orderIsCanceled($event->orderItem->getOrder())) {
			$this->supplyProducts->assignOrderItem($event->orderItem);
		}
	}

	public function orderItemRemoved(OrderItemEvent $event): void
	{
		if (!$this->orderIsCanceled($event->orderItem->getOrder())) {
			$this->supplyProducts->removeOrderItem($event->orderItem);
		}
	}

	public function orderItemSaved(OrderItemEvent $event): void
	{
		if ($this->orderIsCanceled($event->orderItem->getOrder())) {
			return;
		}

		$orderItemId = $event->orderItem->getId();
		if ($orderItemId === null) {
			return;
		}

		$conn   = $this->orderItems->em->getConnection();
		$result = $conn->fetchAssociative('SELECT quantity FROM eshop_orders__order_item WHERE id = ' . $orderItemId . ' LIMIT 1');

		if (!$result) {
			return;
		}

		$oldOrderItemQuantity = $result['quantity'];
		$newOrderItem         = $event->orderItem;
		$changedQuantity      = $newOrderItem->getQuantity();
		$diff                 = abs($newOrderItem->getQuantity() - $oldOrderItemQuantity);

		if ($newOrderItem->getQuantity() > $oldOrderItemQuantity) {
			// doslo k navyseni
			$newOrderItem->setQuantity($diff);
			$this->supplyProducts->assignOrderItem($newOrderItem);
		} else if ($newOrderItem->getQuantity() < $oldOrderItemQuantity) {
			// doslo ke snizeni
			$newOrderItem->setDefaultQuantity($oldOrderItemQuantity);
			$newOrderItem->setQuantity($diff);
			$this->supplyProducts->removeOrderItem($newOrderItem);
		}

		$newOrderItem->setQuantity($changedQuantity);
	}

	protected function orderIsCanceled(Order $order): bool
	{
		$isCanceled = false;
		/** @var OrderStatus $orderStatus */
		foreach ($order->getOrderStatuses()->toArray() as $orderStatus) {
			if ($orderStatus->getStatus()->getId() === OrderStatus::STATUS_CANCELED && !$orderStatus->isDeleted()) {
				$isCanceled = true;
			}
		}

		return $isCanceled;
	}

}
