<?php declare(strict_types = 1);

namespace EshopStock\Model\Repository;

use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Query\Parameter;
use Contributte\Translation\Translator;
use Core\Model\Helpers\BaseEntityService;
use Core\Model\Helpers\Strings;
use Doctrine\ORM\Query\Expr\Join;
use Doctrine\ORM\QueryBuilder;
use EshopCatalog\AdminModule\Model\Products;
use EshopCatalog\Model\Entities\Product;
use EshopCatalog\Model\Entities\ProductListener;
use EshopCatalog\Model\Entities\Supplier;
use EshopOrders\Model\Entities\OrderItem;
use EshopOrders\Model\Entities\OrderStatus;
use EshopStock\Model\Entities\ProductsToOrderOrders;
use EshopStock\Model\Entities\ProductToOrder;
use EshopStock\Model\Entities\Stock;
use EshopStock\Model\Entities\Supply;
use EshopStock\Model\Entities\SupplyProduct;
use Exception;
use Nette\InvalidArgumentException;
use Nette\Utils\DateTime;
use Tracy\Debugger;

/**
 * @method Supply|null get($id)
 */
class Supplies extends BaseEntityService
{
	/** @var string */
	protected $entityClass = Supply::class;

	public function __construct(
		protected Translator      $translator,
		protected ProductsToOrder $productsToOrder,
		protected Products        $products,
		protected SupplyProducts  $supplyProducts,
	)
	{
	}

	public function getQueryBuilder(int $stockId): QueryBuilder
	{
		$qb = $this->em->createQueryBuilder();

		return $qb->select('s, sp')
			->from($this->entityClass, 's')
			->join('s.supplyProducts', 'sp', Join::WITH, 's.id = sp.supply')
			->where('s.stock = :stockId')
			->setParameter('stockId', $stockId);
	}

	public function createQueryFilter(int $stockId): SupplyItemQuery
	{
		return new SupplyItemQuery($this->em, $this->translator, $stockId);
	}

	public function removeSupply(int $id): bool
	{
		$supply = $this->get($id);

		if (!$supply) {
			return false;
		}

		try {
			$this->em->beginTransaction();

			/** @var SupplyProduct[] $supplyProducts */
			$supplyProducts = $supply->getSupplyProducts()->toArray();

			$productsToDelete = [];
			$supplyProductSum = [];
			foreach ($supplyProducts as $supplyProduct) {
				$productId = $supplyProduct->product->getId();
				$productsToDelete[$productId]++;

				// vytvorime si seznam pro kazdy produkt a u neho pocet vyskytu na prijemce a jeho objednavky
				if ($supplyProduct->isWrittenOff() || !$supplyProduct->hasOrder()) {
					continue;
				}
				$orderId = $supplyProduct->order->getId();
				$order   = $supplyProduct->order;
				if (isset($supplyProductSum[$productId])) {
					$supplyProductSum[$productId]['quantity']++;
					$supplyProductSum[$productId]['orders'][$orderId] = $order;
				} else {
					$supplyProductSum[$productId] = ['quantity' => 1, 'orders' => [$orderId => $order]];
				}
			}

			if ($productsToDelete === []) {
				return true;
			}

			foreach ($productsToDelete as $productId => $quantity) {
				/** @var Product|null $product */
				$product = $this->em->getReference(Product::class, (int) $productId);
				if (!$product) {
					continue;
				}
				$product->quantity -= $quantity; // snizeni poctu ks u produktu
				$this->em->persist($product);
			}

			// Zjistime, zda neni produkt v produktech k objednani.
			// 1. Pokud jiz je, zapiseme u ktere objednavky chybi a navysime pocet. (nezapomenout na neduplicitu objednavek)
			// 2. Pokud neni, zalozime novy zaznam (nezapomenout na neduplicitu objednavek)
			foreach ($supplyProducts as $supplyProduct) {
				// pokud je produkt odepsany nebo neni prirazen k objednavce, neni co pripisovat do produktu k objednani
				if ($supplyProduct->isWrittenOff() || !$supplyProduct->hasOrder()) {
					continue;
				}

				// situace, kdy je produkt jiz v produktech k objedani, tak jen pridame chybejici objednavky a navysime pocet
				$productToOrder = $this->productsToOrder->getOneByProduct((int) $productId);
				if ($productToOrder instanceof ProductToOrder) {

					// Projdu objednavky spojene s produktem k objednani
					// 1. Pokud jiz existuje stejna objednavka -> navysime jen pocet
					// 2. Pokud ne, navysime pocet a přidame objednavku

					// V1
					//					$orderFinded = false;
					//					/** @var Order $order */
					//					foreach ($productToOrder->orders->toArray() as $order) {
					//						if ($order->getId() === $supplyProduct->order->getId()) {
					//							$orderFinded = true;
					//							$productToOrder->quantity -= 1; // kvantita je zaporna, -1 -> -2
					//							break;
					//						}
					//					}
					//
					//					if (!$orderFinded) {
					//						$productToOrder->addOrder($supplyProduct->order);
					//						$productToOrder->quantity -= 1;  // kvantita je zaporna, -1 -> -2
					//					}

					// V2 pokud nebude fungovat, zvolit V1
					if (!$productToOrder->orders->contains($supplyProduct->order)) {
						$productToOrder->addOrder($supplyProduct->order);
					}
					$productToOrder->quantity -= 1;  // kvantita je zaporna, -1 -> -2

					$this->em->persist($productToOrder);
				} else { // situace, kdy jeste neni produkt v produktech k objednani, tak ho tam vlozime

					// zalozime novy productToOrder
					$orderId   = $supplyProduct->order->getId();
					$productId = $supplyProduct->product->getId();

					if (isset($supplyProductSum[$productId]['orders'], $supplyProductSum[$productId]['quantity'])) {
						$newProductToOrder = new ProductToOrder($supplyProduct->product, array_values($supplyProductSum[$productId]['orders']), -$supplyProductSum[$productId]['quantity']);
						$this->em->persist($newProductToOrder);
					}
				}
			}

			$this->em->flush();

			$this->remove($id);

			$this->em->commit();
		} catch (Exception) {
			$this->em->rollback();

			return false;
		}

		return true;
	}

	public function createSupply(Stock $stock, Supplier $supplier, string $invoiceNumber, DateTime $dateSupply, array $products): bool
	{
		if ($products === []) {
			throw new InvalidArgumentException('EshopStock: $products is empty');
		}

		ProductListener::$updateVariant = false;
		$this->em->beginTransaction();
		try {
			$supply = new Supply($stock, $supplier, $invoiceNumber, $dateSupply);

			foreach ($products as $key => $val) {
				$productId        = $key;
				$quantity         = $val['quantity'];
				$purchasePrice    = $val['purchasePrice'];
				$recyclingFee     = $val['recyclingFee'] ?? '0';
				$purchasePriceStr = empty($purchasePrice) ? '0' : ((string) $purchasePrice);
				$productToOrder   = $this->productsToOrder->getOneByProduct($productId);

				if ($productToOrder instanceof ProductToOrder) {
					$qb       = $this->em->createQueryBuilder();
					$orderIds = array_map(static fn(ProductsToOrderOrders $productsToOrderOrders): ?int => $productsToOrderOrders->order->getId(), $productToOrder->orders->toArray());

					if ($orderIds === []) { // Rucni dohledani, kde chybí. Vetev kvuli chybe na zurielu, jinak k tomu nikdy nedojde
						$oQb = $this->em->getRepository(OrderItem::class)->createQueryBuilder('oi');
						$oQb->join('oi.order', 'o', 'WITH', 'o.isCorrectiveTaxDocument = 0 AND o.id >= :fromOrder')
							->join('oi.product', 'p', 'WITH', 'p.id = :pId')
							->setParameter('fromOrder', 83600)
							->setParameter('pId', $productId);

						/** @var OrderItem $r */
						foreach ($oQb->getQuery()->getResult() as $r) {
							$statuses = array_map(fn(OrderStatus $os): string => $os->getStatus()->getId(), $r->order->getOrderStatuses()->toArray());
							if (in_array(OrderStatus::STATUS_FINISHED, $statuses) || in_array(OrderStatus::STATUS_CANCELED, $statuses)) {
								continue;
							}

							$assignedOnStockCount = $this->em->getRepository(SupplyProduct::class)->createQueryBuilder('sp')
								->select('COUNT(sp.id)')
								->andWhere('sp.product = :pId AND sp.order = :oId')
								->setParameters(new ArrayCollection([new Parameter('pId', $productId), new Parameter('oId', $r->order->getId())]))
								->getQuery()->getSingleScalarResult();

							if ($assignedOnStockCount < $r->getQuantity()) {
								$orderIds[] = $r->getOrder()->getId();
							}
						}
					}

					if ($orderIds !== []) {

						$qb->select('oi')
							->from(OrderItem::class, 'oi')
							->join('oi.order', 'o')
							->join('oi.product', 'p')
							->andWhere($qb->expr()->in('o.id', $orderIds))
							->andWhere($qb->expr()->eq('p.id', $productId))
							->orderBy('o.id');

						/** @var OrderItem[] $orderItems */
						$orderItems = $qb->getQuery()->getResult();

						foreach ($orderItems as $item) {
							$order                 = $item->order;
							$orderId               = $item->order->getId();
							$stockedProductsCount  = count($this->supplyProducts->getProductsByOrder($orderId, $productId, $stock->getId())); // kusy, ktere uz jsou prirazeny
							$missingProductsCount  = $item->getQuantity() - $stockedProductsCount; // skutecne chybejici kusy u objednavky
							$removeProductsToOrder = $quantity >= $missingProductsCount;
							$productsToOrder       = min($quantity, $missingProductsCount);

							// snizime naskladnovany pocet ks o to, co je v polozce objednavky produktu, ale jen ty kusy, co opravdu chybi (nektere predchozi uz mohou byt prirazene)
							$quantity -= $productsToOrder;

							// prirazeni objednavky k naskladnovanemu produktu
							for ($i = 0; $i < $productsToOrder; $i++) {
								$supplyProduct        = new SupplyProduct($this->products->get($productId), $supply, $purchasePriceStr);
								$supplyProduct->order = $order;
								$supply->addSupplyProduct($supplyProduct);
							}

							// snizime počet ks u chybejiciho produktu
							$productToOrder->quantity += $productsToOrder;

							if ($removeProductsToOrder) {
								// bylo naskladneny vsechny kusy produktu, ktere chybely, odebereme objednavku od chybejiciho produktu
								/** @var ProductsToOrderOrders $o */
								foreach ($productToOrder->orders->toArray() as $o) {
									if ($o->order->getId() === $orderId) {
										$productToOrder->orders->removeElement($o);
										$this->em->remove($o);
									}
								}
							}
						}

						$this->em->persist($productToOrder);
					}
				}

				$product = $this->products->get($productId);

				// naskladneni produktu bez objednavek
				for ($i = 0; $i < $quantity; $i++) {
					$supply->addSupplyProduct(new SupplyProduct($product, $supply, $purchasePriceStr));
				}

				// zvyseni kvantity o pocet (ktery jiz nevykryva objednavky) pri naskladneni
				$product->quantity += (int) $val['quantity'];
				// aktualizace porizovaci ceny, aby se aktualizovala marže v katalogu
				$product->purchasePrice = empty($purchasePrice) ? '0' : Strings::formatEntityDecimal($purchasePrice);

				$product->recyclingFee = empty($recyclingFee) ? '0' : Strings::formatEntityDecimal($recyclingFee);

				$this->em->persist($product);
			}

			$this->em->persist($supply);
			$this->em->flush();

			$this->em->commit();
		} catch (Exception $exception) {
			Debugger::log($exception);
			$this->em->rollback();

			return false;
		}

		return true;
	}

}
