<?php declare(strict_types = 1);

namespace Essox\Model;

use Core\Model\Helpers\Arrays;
use EshopOrders\Model\Entities\Order;
use libphonenumber\PhoneNumberUtil;
use Nette\Caching\Cache;
use Nette\Caching\Storage;
use Nette\Utils\Json;

class ApiService
{
	protected bool    $allowed;
	protected bool    $isTest;
	protected ?string $clientKey    = null;
	protected ?string $clientSecret = null;

	protected string $prodEndPoint = 'https://apiv32.essox.cz';
	protected string $testEndPoint = 'https://testapiv32.essox.cz';

	protected Cache $cache;

	public function __construct(
		bool    $allowed,
		bool    $isTest,
		?string $clientKey,
		?string $clientSecret,
		Storage $cacheStorage
	)
	{
		$this->allowed      = $allowed;
		$this->isTest       = $isTest;
		$this->clientKey    = $clientKey;
		$this->clientSecret = $clientSecret;

		$this->cache = new Cache($cacheStorage, 'essox');
	}

	public function isAllowed(): bool
	{
		return $this->allowed && $this->clientKey && $this->clientSecret;
	}

	protected function getEndPoint(): string
	{
		return $this->isTest ? $this->testEndPoint : $this->prodEndPoint;
	}

	protected function getToken(): ?string
	{
		$cacheKey = 'token/' . md5(implode('|', [$this->isTest, $this->clientKey, $this->clientSecret]));

		$result = $this->cache->load($cacheKey) ?: null;

		if ($result) {
			return $result;
		}

		$ch = curl_init();
		curl_setopt($ch, CURLOPT_URL, $this->getEndPoint() . '/token');
		curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
		curl_setopt($ch, CURLOPT_TIMEOUT, 0);
		curl_setopt($ch, CURLOPT_FOLLOWLOCATION, 1);
		curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'POST');
		curl_setopt($ch, CURLOPT_HTTPHEADER, [
			'Content-Type: application/json',
			'Authorization: Basic ' . base64_encode($this->clientKey . ':' . $this->clientSecret),
		]);
		curl_setopt($ch, CURLOPT_POSTFIELDS, Json::encode([
			'grant_type' => 'client_credentials',
			'scope'      => 'scopeFinit.consumerGoods.eshop',
		]));

		$response = curl_exec($ch);
		curl_close($ch);

		if (Arrays::isJson($response)) {
			$data = Json::decode($response, Json::FORCE_ARRAY);

			$expire = $data['expires_in'] - 60;
			$result = $data['access_token'];

			if ($result) {
				$this->cache->save($cacheKey, $result, [Cache::EXPIRE => $expire . ' seconds']);
			}
		}

		return $result;
	}

	public function getUrl(float $price, ?int $productId): ?string
	{
		$result = null;

		$token = $this->getToken();
		if (!$token) {
			return null;
		}

		$ch = curl_init();
		curl_setopt($ch, CURLOPT_URL, $this->getEndPoint() . '/consumergoods/v1/api/consumergoods/calculator');
		curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
		curl_setopt($ch, CURLOPT_TIMEOUT, 0);
		curl_setopt($ch, CURLOPT_FOLLOWLOCATION, 1);
		curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'POST');
		curl_setopt($ch, CURLOPT_HTTPHEADER, [
			'Content-Type: application/json',
			'Authorization: Bearer ' . $token,
		]);

		curl_setopt($ch, CURLOPT_POSTFIELDS, Json::encode([
			'price'     => $price,
			'productId' => $productId ?: 0,
		]));

		$response = curl_exec($ch);
		curl_close($ch);

		if (Arrays::isJson($response)) {
			$data = Json::decode($response, Json::FORCE_ARRAY);

			if (isset($data['redirectionUrl'])) {
				$result = $data['redirectionUrl'];
			}
		}

		return $result;
	}

	public function getPaymentUrl(Order $order, string $returnLink): ?string
	{
		$result = null;

		$token = $this->getToken();
		if (!$token) {
			return null;
		}

		$ch = curl_init();
		curl_setopt($ch, CURLOPT_URL, $this->getEndPoint() . '/consumergoods/v1/api/consumergoods/proposal');
		curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
		curl_setopt($ch, CURLOPT_TIMEOUT, 0);
		curl_setopt($ch, CURLOPT_MAXREDIRS, 10);
		curl_setopt($ch, CURLOPT_FOLLOWLOCATION, 1);
		curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'POST');
		curl_setopt($ch, CURLOPT_HTTPHEADER, [
			'Content-Type: application/json',
			'Authorization: Bearer ' . $token,
		]);

		$invoiceAddr = $order->getAddressInvoice();
		if (!$invoiceAddr) {
			return null;
		}

		$parsedPhone = null;
		if ($invoiceAddr->getPhone() && $invoiceAddr->getCountry()) {
			$phoneUtil   = PhoneNumberUtil::getInstance();
			$parsedPhone = $phoneUtil->parse($invoiceAddr->getPhone(), $invoiceAddr->getCountry()->getId());
		}

		curl_setopt($ch, CURLOPT_POSTFIELDS, Json::encode([
			'firstName'           => $invoiceAddr->getFirstName(),
			'lastName'            => $invoiceAddr->getLastName(),
			'mobilePhonePrefix'   => $parsedPhone ? '+' . $parsedPhone->getCountryCode() : '',
			'mobilePhoneNumber'   => $parsedPhone ? $parsedPhone->getNationalNumber() : '',
			'email'               => $invoiceAddr->getEmail(),
			'productId'           => 0,
			'price'               => $order->getPrice(),
			'orderId'             => (string) $order->getId(),
			'customerId'          => $order->getCustomer() ? (string) $order->getCustomer()->getId() : null,
			'callbackUrl'         => $returnLink,
			'spreadedInstalments' => false,
		]));

		$response = curl_exec($ch);
		curl_close($ch);

		if (Arrays::isJson($response)) {
			$data = Json::decode($response, Json::FORCE_ARRAY);

			if (isset($data['redirectionUrl'])) {
				$result = $data['redirectionUrl'];
			}
		}

		return $result;
	}
}
