<?php declare(strict_types=1);

namespace Fio\AdminModule\Model\Subscribers;

use EshopPayments\AdminModule\Model\Events\PaymentSyncEvent;
use EshopPayments\Model\Facade\PaymentsFacade;
use Fio\AdminModule\Model\IFioDownloaderFactory;
use Fio\Model\Entities\FioPayment;
use Fio\Model\FioConfig;
use Fio\Model\FioDownloader;
use Nette\Utils\Arrays;
use Nette\Utils\DateTime;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Tracy\Debugger;

class SyncSubscriber implements EventSubscriberInterface
{
	protected string $bankCode = '2010';
	protected IFioDownloaderFactory $fioDownloaderFactory;

	public function __construct(IFioDownloaderFactory $fioDownloaderFactory)
	{
		$this->fioDownloaderFactory = $fioDownloaderFactory;
	}

	public static function getSubscribedEvents()
	{
		return [
			PaymentsFacade::class . '::onSync' => ['onSync', 100],
		];
	}

	public function onSync(PaymentSyncEvent $event): void
	{
		$payment = $event->getSellerBankAccount();
		$lastSyncedPayment = $event->getLastSyncedPayment();

		// overeni, zda ucet spada pod Fio banku
		if ($payment->numberPart2 !== $this->bankCode) {
			return;
		}

		// rozhodnuti, jake bude datum, od ktereho se budou importovat platby z banky
		if ($lastSyncedPayment) {
			// datum posledního vlozeneho zaznamu
			$dateSince = $lastSyncedPayment->inserted;
		} else {
			// datum z configu
			/** @var \DateTimeInterface $dateSince */
			$dateSince = DateTime::createFromFormat('Y-m-d', FioConfig::load('syncDateFrom'));
		}

		try {
			$downloader = $this->fioDownloaderFactory->create($payment->numberPart1);
			$transactionList = $downloader->downloadSince($dateSince);

			foreach ($transactionList->getTransactions() as $transaction) {
				$p = new FioPayment($transaction->getAmount(), (string) $transaction->getPaymentOrderId(), $payment);
				$p->setDateOfPayment($transaction->getDate())
				  ->setSenderAccountNumber($transaction->getSenderAccountNumber())
				  ->setSenderBankCode($transaction->getSenderBankCode())
				  ->setSenderBankName($transaction->getSenderBankName())
				  ->setSenderName($transaction->getSenderName())
				  ->setConstantSymbol($transaction->getConstantSymbol())
				  ->setSpecificSymbol($transaction->getSpecificSymbol())
				  ->setVariableSymbol($transaction->getVariableSymbol())
				  ->setUserMessage($transaction->getUserMessage())
				  ->setComment($transaction->getComment());

				$event->paymentsToSync[] = $p;
			}

		} catch (\Exception $ex) {
			Debugger::log($ex, 'fio');
		}
	}

}