<?php declare(strict_types = 1);

namespace Fio\DI;

use Core\DI\CompilerExtension;
use Nette;
use Nette\DI\Compiler;
use Nette\Schema\Expect;
use Nette\Schema\Schema;
use Nette\Utils\DateTime;

class FioExtension extends CompilerExtension
{
	public function getConfigSchema(): Schema
	{
		$data = $this->getContainerBuilder()->parameters['fio'];
		$schema = Expect::structure([
			'accounts' => Expect::arrayOf(
				Expect::structure([
					'account' => Expect::string()->required(),
					'token' => Expect::string()->required(),
				]
			)),
			'syncDateFrom' => Expect::string()->required()->assert(static function ($val) {
				if ($val instanceof DateTime) {
					return true;
				}

				try {
					new DateTime($val);
					return true;
				} catch (\Exception $e) {
					return false;
				}
			}, 'Please define parameter syncDateFrom in correct format (Y-m-d)')
		]);
		(new Nette\Schema\Processor())->process($schema, $data);

		if (!Nette\Utils\Arrays::every(array_keys($data['accounts']), static function ($val) {
			if (Nette\Utils\Strings::contains($val, '-') && count(explode('-', $val)) === 2) {
				[$curr, $permission] = explode('-', $val);
				if (strlen($curr) === 3 && in_array($permission, ['read', 'write'])) {
					return true;
				}
			}

			return false;
		})) {
			throw new Nette\Schema\ValidationException('Keys in accounts must be in format currency{3chars}-permission{read|write} e.g. czk-read or czk-write');
		}

		return parent::getConfigSchema();
	}

	public function loadConfiguration()
	{
		parent::loadConfiguration();
		$this->setConfig($this->loadFromFile(__DIR__ . '/config.neon'));

		$this->compiler->loadDefinitionsFromConfig($this->loadFromFile(__DIR__ . '/services.neon'));
	}

	public function beforeCompile()
	{
		parent::beforeCompile();
		$this->setMapping(['Fio' => 'Fio\*Module\Presenters\*Presenter']);
	}

	public function afterCompile(Nette\PhpGenerator\ClassType $class)
	{
		parent::afterCompile($class);

		$init    = $class->methods["initialize"];
		$builder = $this->getContainerBuilder();

		$init->addBody('Fio\Model\FioConfig::setParams($this->parameters[\'fio\']);');
	}
}
