<?php declare(strict_types = 1);

namespace Forms\AdminModule\Components;

use Core\Model\Helpers\FormHelper;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Forms\AdminModule\Model\Forms;
use Forms\Model\Entities\FormMeasuringCodes;
use Forms\Model\Entities\FormText;
use Forms\Model\FormsCache;
use MeasuringCodes\DI\MeasuringCodesExtension;
use Nette\Caching\Cache;
use Nette\InvalidArgumentException;
use Nette\Utils\ArrayHash;
use Forms\Model\Entities\Form;

class FormForm extends BaseControl
{
	/** @var int|null @persistent */
	public ?int $formId = null;

	protected Forms      $forms;
	protected FormsCache $formsCache;

	protected array $userTextFields = ['userEmailSubject', 'userFromEmail', 'userFromName', 'userToEmail', 'userToName',
		'userBc', 'userBcc', 'userReply'];

	protected array $adminTextFields = ['adminEmailSubject', 'adminFromEmail', 'adminFromName', 'adminToEmail',
		'adminToName', 'adminReply'];

	public function __construct(
		Forms      $forms,
		FormsCache $formsCache
	)
	{
		$this->forms      = $forms;
		$this->formsCache = $formsCache;
	}

	public function render(): void
	{
		$this->template->userTextFields  = $this->userTextFields;
		$this->template->adminTextFields = $this->adminTextFields;
		$this->template->thisForm        = $this['form'];
		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAjax();

		$form->addText('title', 'forms.form.title')
			->setIsMultilanguage();
		$form->addText('ident', 'forms.form.ident')
			->setRequired();
		$form->addText('subject', 'forms.form.subject')
			->setIsMultilanguage();
		$form->addEditor('messageAfterSend', 'forms.form.messageAfterSend')
			->setDisableAutoP(false)
			->setHeight(100)
			->setIsMultilanguage();
		$form->addBool('isPublished', 'default.isPublished')
			->setDefaultValue(1);

		foreach (array_merge($this->userTextFields, $this->adminTextFields) as $col) {
			$form->addText($col, 'forms.form.' . $col);
		}

		$form['userEmailSubject']->setIsMultiLanguage();
		$form['adminEmailSubject']->setIsMultiLanguage();

		$templates = [];
		foreach (glob(PACKAGE_FORMS_DIR . '/FrontModule/Templates/EmailTemplates/*.latte') as $file) {
			$fileName             = pathinfo($file, PATHINFO_FILENAME);
			$templates[$fileName] = 'forms.templates.' . $fileName;
		}
		foreach (glob(TEMPLATES_DIR . '/Front/default/Forms/EmailTemplates/*.latte') as $file) {
			$fileName             = pathinfo($file, PATHINFO_FILENAME);
			$templates[$fileName] = 'forms.templates.' . $fileName;
		}
		$form->addSelect('template', 'forms.form.template', $templates);

		$form->addCheckbox('sendUserEmail', 'forms.form.sendUserEmail')
			->setDefaultValue(0);
		$form->addCheckbox('sendUserEmailConfirm', 'forms.form.sendUserEmailConfirm')
			->setDefaultValue(0);
		$form->addCheckbox('sendAdminEmail', 'forms.form.sendAdminEmail')
			->setDefaultValue(0);
		$form->addEditor('userEmailContent', 'forms.form.userEmailContent')
			->setIsMultilanguage();
		$form->addEditor('userEmailConfirm', 'forms.form.userEmailConfirm')
			->setIsMultilanguage();
		$form->addEditor('adminEmailContent', 'forms.form.adminEmailContent')
			->setIsMultilanguage();

		if (class_exists(MeasuringCodesExtension::class)) {
			$container = $form->addContainer('measuringCodes');
			$container->addText('googleAdsConversionLabel', 'forms.measuringCodes.googleAdsConversionLabel');
			$container->addText('sklikConversionId', 'forms.measuringCodes.sklikConversionId');
		}

		$form->addSaveCancelControl();

		$form->onValidate[] = [$this, 'formValidate'];
		$form->onSuccess[]  = [$this, 'formSuccess'];

		return $form;
	}

	public function formValidate(BaseForm $form, ArrayHash $values): bool
	{
		if (!$this->forms->validateFormIdent($values->ident, $this->formId)) {
			$form->addError('forms.form.identExist');
			$this->redrawControl('form');

			return false;
		}

		return true;
	}

	public function formSuccess(BaseForm $baseForm, ArrayHash $values): bool
	{
		try {
			$langValues = $baseForm->convertMultilangValuesToArray();

			if ($this->formId) {
				$form  = $this->forms->get($this->formId);
				$texts = $form->getTexts()->toArray();
			} else {
				$form  = new Form($values->ident);
				$texts = [];
			}

			foreach ($langValues as $l => $v) {
				if (!isset($texts[$l]))
					$texts[$l] = new FormText($form, $l, $v['title']);

				$texts[$l]->title            = $v['title'];
				$texts[$l]->messageAfterSend = $v['messageAfterSend'] ?: null;

				foreach (['userEmailSubject', 'userEmailContent', 'adminEmailSubject', 'adminEmailContent'] as $col) {
					$texts[$l]->$col = $v[$col];
					unset($values->$col);
				}

				$texts[$l]->userEmailConfirm = $v['userEmailConfirm'];
				unset($values->userEmailConfirm);

				$this->em->persist($texts[$l]);
				$form->texts->set($l, $texts[$l]);
			}

			foreach (['sendUserEmail', 'sendUserEmailConfirm', 'sendAdminEmail'] as $col) {
				$form->$col = (int) $values->$col;
				unset($values->$col);
			}

			$form->setIdent($values->ident);
			$this->em->persist($form);

			// TODO multijazykove
			if (class_exists(MeasuringCodesExtension::class)) {
				$currentMS = $form->getMeasuringCodesByKey();

				foreach ($values->measuringCodes as $k => $v) {
					if (!isset($currentMS[$k])) {
						$ms = new FormMeasuringCodes($form, $k);
						$form->measuringCodes->add($ms);
					} else {
						$ms = $currentMS[$k];
					}

					$ms->value = $v;

					$this->em->persist($ms);
				}

				// delete
				foreach (array_diff_key($currentMS, (array) $values->measuringCodes) as $v) {
					$this->em->remove($v);
				}

				unset($values->measuringCodes);
			}

			FormHelper::fillEntityByValues($form, $values);
			$this->em->persist($form);

			$this->em->flush();

			$this->getPresenter()->flashMessageSuccess('default.saved');
			$this->getPresenter()->redrawControl('flashes');

			$baseForm->addCustomData('formId', $form->getId());

			$this->formsCache->getCache()->remove('form_' . $form->getId());
			$this->formsCache->getCache()->remove('form_' . $form->getIdent());

			return true;
		} catch (\Exception $e) {
			$this->getPresenter()->flashMessageDanger('default.error');
			$this->getPresenter()->redrawControl('flashes');
		}

		return false;
	}

	public function setForm(int $formId): void
	{
		$this->formId = $formId;
		$form         = $this->forms->get($formId);

		if (!$form)
			throw new InvalidArgumentException();

		FormHelper::fillContainerByEntity($this['form'], $form);

		if (class_exists(MeasuringCodesExtension::class)) {
			$ms = $form->getMeasuringCodesByKey();
			foreach ($this['form']['measuringCodes']->getComponents() as $c) {
				$c->setDefaultValue($ms[$c->getName()] ? $ms[$c->getName()]->value : null);
			}
		}

		$texts = [
			'ident' => $form->getIdent(),
		];
		foreach ($form->getTexts()->toArray() as $l => $e) {
			/** @var FormText $e */
			$texts['title'][$l]            = $e->title;
			$texts['messageAfterSend'][$l] = $e->messageAfterSend;
			foreach (['userEmailSubject', 'userEmailContent', 'userEmailConfirm', 'adminEmailSubject',
				         'adminEmailContent'] as $col)
				$texts[$col][$l] = $e->$col;
		}

		$this['form']->setDefaults($texts);
	}
}
