<?php declare(strict_types = 1);

namespace Forms\AdminModule\Components;

use Core\Model\Entities\QueryBuilder;
use Core\Model\Helpers\Arrays;
use Core\Model\Helpers\Strings;
use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use Forms\AdminModule\Model\Submissions;
use Forms\Model\Entities\Submission;
use Forms\Model\Entities\SubmissionField;
use Nette\Utils\Html;
use Nette\Utils\Json;
use Nette\Utils\Validators;

class FormSubmissions extends BaseControl
{
	/** @var int @persistent */
	public int $formId;

	protected Submissions $submissions;

	public function __construct(int $formId, Submissions $submissions)
	{
		$this->formId      = $formId;
		$this->submissions = $submissions;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();

		$qb = $this->submissions->getEr()->createQueryBuilder('s')
			->where('s.form = :form')
			->leftJoin('s.fields', 'sf')
			->setParameter('form', $this->formId)
			->orderBy('s.created', 'DESC');

		$grid->setDataSource($qb);

		$submissionValues                      = [];
		$grid->getDataSource()->onDataLoaded[] = function($items) use (&$submissionValues) {
			$ids = array_map(function($row) { return $row->getId(); }, $items);

			foreach ($this->em->getRepository(SubmissionField::class)->createQueryBuilder('sf')
				         ->addSelect('IDENTITY(sf.submission) as submission')
				         ->where('sf.submission IN (:ids)')->setParameter('ids', $ids)
				         ->getQuery()->getArrayResult() as $row) {
				$submissionValues[$row['submission']][$row[0]['name']] = $row[0]['value'];
			}
		};

		$grid->addColumnNumber('id', 'forms.grid.id');
		foreach ($this->submissions->getFieldKeys($this->formId) as $name) {
			$grid->addColumnText($name, $name)
				->setRenderer(function(Submission $row) use ($name, &$submissionValues) {
					$value = $submissionValues[$row->getId()][$name] ?? null;

					$createHtml = static function($v): Html {
						$wrap = Html::el('');

						if (Validators::isUrl($v) || Strings::startsWith($v, '//')) {
							$el = Html::el('div')
								->addHtml(Html::el('a', [
									'href'   => $v,
									'target' => '_blank',
								])
									->setText(basename($v)));
						} else {
							$el = Html::el('div')
								->setText($v);
						}
						$wrap->addHtml($el);

						return $wrap;
					};

					if (Arrays::isJson($value) && !is_numeric($value)) {
						$wrap = Html::el('');

						foreach (Json::decode($value) as $v) {
							$wrap->addHtml($createHtml($v));
						}

						return $wrap;
					} else {
						$value = $createHtml((string) $value);
					}

					return $value;
				})->setFilterText()->setCondition(function(QueryBuilder $qb, $value) use ($name) {
					$qb->andWhere("sf.name = :{$name}Key AND sf.value LIKE :{$name}Val")
						->setParameter($name . 'Key', $name)
						->setParameter($name . 'Val', "%{$value}%");
				});
		}

		$grid->addColumnText('confirmed', 'forms.submission.confirmed')->setRenderer(function(Submission $row) {
			return $row->confirmed ? '1' : '0';
		})->setFilterText();
		$grid->addColumnDateTime('created', 'forms.submission.created')->setFormat('j. n. Y H:i:s');

		return $grid;
	}
}
