<?php declare(strict_types = 1);

namespace Forms\AdminModule\Components;

use Core\Model\Http\CsvResponse;
use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use Doctrine\ORM\Query\Expr\Join;
use Forms\AdminModule\Model\Forms;
use Forms\AdminModule\Model\Submissions;
use Forms\Model\Entities\Form;
use Forms\Model\FormsCache;
use Nette\Caching\Cache;
use Nette\Utils\Html;

class FormsGrid extends BaseControl
{
	protected IFormFormFactory        $formFormFactory;
	protected Submissions             $submissions;
	protected Forms                   $forms;
	protected IFormSubmissionsFactory $formSubmissionsFactory;
	protected FormsCache              $formsCache;

	public function __construct(
		IFormFormFactory        $formFactory,
		Submissions             $submissions,
		Forms                   $forms,
		IFormSubmissionsFactory $formSubmissionsFactory,
		FormsCache              $formsCache
	)
	{
		$this->formFormFactory        = $formFactory;
		$this->submissions            = $submissions;
		$this->forms                  = $forms;
		$this->formSubmissionsFactory = $formSubmissionsFactory;
		$this->formsCache             = $formsCache;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ======================== Handle
	 */

	public function handleAdd(): void
	{
		$this->template->modalTitle       = $this->t('forms.title.newForm');
		$this->template->modalDialogClass = 'modal-xl';
		$this->template->modal            = 'formForm';
		$this->redrawControl('modal');
	}

	public function handleEdit(int $id): void
	{
		$this->template->modalTitle = $this->t('forms.title.editForm');
		$this['form']->setForm($id);
		$this->template->modalDialogClass = 'modal-xl';
		$this->template->modal            = 'formForm';
		$this->redrawControl('modal');
	}

	public function handleDelete(int $id): void
	{
		$p = $this->getPresenter();
		if ($this->forms->remove($id)) {
			$p->flashMessageSuccess('default.removed');
			$this['grid']->reload();
		} else
			$p->flashMessageDanger('default.removeFailed');

		$p->redrawControl('flashes');
	}

	public function handleSubmissions(int $id): void
	{
		$this['submissionsGrid']->formId  = $id;
		$this->template->modal            = 'submissions';
		$this->template->modalDialogClass = 'modal-xxl';
		$this->redrawControl('modal');
	}

	public function handleExport(int $id): void
	{
		$csv = $this->submissions->exportCsv($id);

		CsvResponse::sendResponse('form-' . $id . '.csv', $csv);
	}

	/*******************************************************************************************************************
	 * ======================== Components
	 */

	public function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();

		$submissionsCount = $this->submissions->getAllCount();

		$qb = $this->em->getRepository(Form::class)->createQueryBuilder('f')
			->addSelect('ft')
			->innerJoin('f.texts', 'ft', Join::WITH, 'ft.lang = :lang')
			->setParameters([
				'lang' => $this->translator->getLocale(),
			]);
		$grid->setDataSource($qb);

		$grid->addColumnText('id', 'forms.grid.id')->setRenderer(function(Form $row) {
			return $row->getId() . ' | ' . $row->getIdent();
		});
		$grid->addColumnText('title', 'forms.grid.title')
			->setRenderer(fn(Form $form) => $form->getText()->title);
		$grid->addColumnNumber('submissions', 'forms.form.submissionsCount')
			->setRenderer(function(Form $form) use ($submissionsCount) {
				$html = Html::el();
				$html->addHtml(Html::el('span')->setText($submissionsCount[$form->getId()] ?? 0));
				$html->addHtml(
					Html::el('a', ['href'  => $this->link('submissions!', $form->getId()), 'target' => '_blank',
					               'class' => 'btn btn-xs btn-primary ml-3 ajax'])
						->addHtml(Html::el('span', ['class' => 'fas fa-eye']))
				);
				$html->addHtml(
					Html::el('a', ['href'  => $this->link('export!', $form->getId()), 'target' => '_blank',
					               'class' => 'btn btn-xs btn-primary ml-1'])
						->addHtml(Html::el('span', ['class' => 'fas fa-file-export']))
				);

				return $html;
			});
		$grid->addColumnStatus('isPublished', 'default.isPublished')->setAlign('center')
			->addOption(1, 'default.publish')->setIcon('check')->setClass('btn-success')->setShowTitle(false)->endOption()
			->addOption(0, 'default.unPublish')->setIcon('times')->setClass('btn-danger')->setShowTitle(false)->endOption()
			->onChange[] = [$this, 'gridPublishChange'];

		// Actions
		$grid->addAction('edit', '', 'edit!')->setIcon('edit')->addClass('ajax')->setBsType('primary');
		$grid->addAction('delete', '', 'delete!')->setIcon('times')->setBsType('danger')->addClass('ajax')
			->setConfirm('default.reallyDelete');

		$grid->getColumn('id')->getElementPrototype('td')->class[] = 'w1nw';

		return $grid;
	}

	protected function createComponentForm(): FormForm
	{
		$control = $this->formFormFactory->create();

		$control->onAnchor[] = function() use ($control) {
			$control['form']->onSuccessSave[] = function() {
				$this->getPresenter()->redrawControl('flashes');
				$this['grid']->reload();
			};

			$control['form']->onSuccessSaveAndClose[] = function() {
				$this->getPresenter()->payload->hideModal = true;
				$this->getPresenter()->redrawControl('flashes');
				$this['grid']->reload();
			};

			$control['form']['saveControl']->closeModalOnCancel();
		};

		return $control;
	}

	public function gridPublishChange($id, $newStatus)
	{
		$presenter = $this->getPresenter();

		if ($this->forms->setPublish($id, $newStatus)) {
			$presenter->flashMessageSuccess('default.publishChanged');
			$this->formsCache->getCache()->clean([
				Cache::Tags => 'form',
			]);
		} else {
			$presenter->flashMessageDanger('default.publishChangeFailed');
		}

		if ($presenter->isAjax()) {
			$this['grid']->redrawItem($id);
			$presenter->redrawControl('flashes');
		} else
			$presenter->redirect('this');
	}

	public function createComponentSubmissionsGrid(): FormSubmissions
	{
		return $this->formSubmissionsFactory->create((int) $this->getParameter('id'));
	}
}
