<?php declare(strict_types = 1);

namespace Forms\FrontModule\Components;

use Core\Model\BotDetect;
use Core\Model\Event\ComponentTemplateEvent;
use Core\Model\Event\CreateFormEvent;
use Core\Model\Mailing\MailBuilderFactory;
use Core\Model\Sites;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Forms\FrontModule\Model\FormProcess;
use Forms\FrontModule\Model\Forms;
use Forms\FrontModule\Model\Dao;
use Nette\Http\Request;
use Nette\Utils\ArrayHash;

class FormControl extends BaseControl
{
	public ?string $templateFile = null;

	/** @var Dao\Form|null */
	protected $form = null;

	protected Forms              $forms;
	protected Request            $request;
	protected Sites              $sites;
	protected MailBuilderFactory $mailBuilder;
	protected FormProcess        $formProcess;

	public function __construct(
		$formId,
		Forms $forms,
		Request $request,
		Sites $sites,
		MailBuilderFactory $mailBuilder,
		FormProcess $formProcess
	)
	{
		$this->forms       = $forms;
		$this->request     = $request;
		$this->sites       = $sites;
		$this->mailBuilder = $mailBuilder;
		$this->form        = $this->forms->getBasicData($formId);
		$this->formProcess = $formProcess;
	}

	public function render(): void
	{
		if (!$this->form)
			return;

		$this->template->thisForm = $this['form'];
		$this->template->formDao  = $this->form;

		$file = TEMPLATES_DIR . '/Front/default/Forms/EmailTemplates/' . $this->form->template . '.latte';
		if (!file_exists($file))
			$file = PACKAGE_FORMS_DIR . '/FrontModule/Templates/EmailTemplates/' . $this->form->template . '.latte';

		$event                  = new ComponentTemplateEvent($this->template, $this);
		$event->data['formDao'] = $this->form;
		$this->eventDispatcher->dispatch($event, FormControl::class . '::render');
		$this->eventDispatcher->dispatch($event, FormControl::class . '::render' . ucfirst($this->form->template));

		$this->template->render($file);
	}

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAction($this->request->getUrl()->getAbsoluteUrl());

		$event                  = new CreateFormEvent($form, $this->template);
		$event->control         = $this;
		$event->data['formDao'] = $this->form;
		$this->eventDispatcher->dispatch($event, FormControl::class . '::create');
		$this->eventDispatcher->dispatch($event, FormControl::class . '::create' . ucfirst($this->form->template));

		$form->onValidate[] = [$this, 'formValidate'];
		$form->onSuccess[]  = [$this, 'formSuccess'];

		return $form;
	}

	public function formValidate(BaseForm $form, ArrayHash $values)
	{
		if ($form->hasErrors())
			$this->redrawControl('formErrors');
	}

	public function formSuccess(BaseForm $baseForm, ArrayHash $values): bool
	{
		if (BotDetect::isBot()) {
			$this->redrawControl('form');

			return true;
		}

		$form = $this->forms->get($this->form->ident);
		if (!$form) {
			$baseForm->addError('formsFront.sendingError');
			$this->redrawControl('formErrors');

			return false;
		}

		$submission = $this->formProcess->createSubmission(
			$values,
			$form,
			$baseForm,
			$this->getPresenter(false) ? $this->getPresenter() : null
		);

		try {
			if ($submission) {
				$this->formProcess->sendUser(
					$values,
					$form,
					$submission,
					$baseForm
				);

				$this->formProcess->sendAdmin(
					$values,
					$form,
					$baseForm
				);

				$this->template->formSent        = true;
				$this->template->thankYouMessage = $form->getText()->messageAfterSend;
				$this->template->formEntity      = $form;

				$this->formProcess->formSuccess(
					$values,
					$form,
					$submission,
					$baseForm,
					$this->getPresenter(false) ? $this->getPresenter() : null
				);

				$this->redrawControl('form');
			} else {
				$baseForm->addError('formsFront.sendingError');
				$this->redrawControl('formErrors');

				return false;
			}
		} catch (\Exception $e) {
			$baseForm->addError('formsFront.sendingError');
			$this->redrawControl('formErrors');

			return false;
		}

		return true;
	}
}
