<?php declare(strict_types = 1);

namespace Forms\Model\Entities;

use Core\Model\Entities\TId;
use Core\Model\Entities\TTranslateListener;
use Core\Model\Helpers\Strings;
use Doctrine\Common\Collections\ArrayCollection;
use Gedmo\Mapping\Annotation as Gedmo;
use Doctrine\ORM\Mapping as ORM;

/**
 * @ORM\Table(name="forms__form")
 * @ORM\Entity
 * @ORM\EntityListeners({"Core\Model\Entities\TranslateListener"})
 */
class Form
{
	use TId;
	use TTranslateListener;

	/**
	 * @ORM\Column(name="ident", type="string", length=255, nullable=true)
	 */
	protected ?string $ident = null;

	/**
	 * @var FormText[]
	 * @ORM\OneToMany(targetEntity="FormText", mappedBy="form", indexBy="lang")
	 */
	public $texts;

	/**
	 * @ORM\Column(name="template", type="string", length=255, nullable=true)
	 */
	public ?string $template = null;

	/**
	 * @ORM\Column(name="is_published", type="smallint", options={"default":1})
	 */
	public int $isPublished = 1;

	/**
	 * @ORM\Column(name="user_from_email", type="string", length=255, nullable=true)
	 */
	public ?string $userFromEmail = null;

	/**
	 * @ORM\Column(name="user_from_name", type="string", length=255, nullable=true)
	 */
	public ?string $userFromName = null;

	/**
	 * @ORM\Column(name="user_to_email", type="string", length=255, nullable=true)
	 */
	public ?string $userToEmail = null;

	/**
	 * @ORM\Column(name="user_to_name", type="string", length=255, nullable=true)
	 */
	public ?string $userToName = null;

	/**
	 * @ORM\Column(name="user_bc", type="string", length=255, nullable=true)
	 */
	public ?string $userBc = null;

	/**
	 * @ORM\Column(name="user_bcc", type="string", length=255, nullable=true)
	 */
	public ?string $userBcc = null;

	/**
	 * @ORM\Column(name="user_reply", type="string", length=255, nullable=true)
	 */
	public ?string $userReply = null;

	/**
	 * @ORM\Column(name="admin_from_email", type="string", length=255, nullable=true)
	 */
	public ?string $adminFromEmail = null;

	/**
	 * @ORM\Column(name="admin_from_name", type="string", length=255, nullable=true)
	 */
	public ?string $adminFromName = null;

	/**
	 * @ORM\Column(name="admin_to_email", type="string", length=255, nullable=true)
	 */
	public ?string $adminToEmail = null;

	/**
	 * @ORM\Column(name="admin_to_name", type="string", length=255, nullable=true)
	 */
	public ?string $adminToName = null;

	/**
	 * @ORM\Column(name="admin_reply", type="string", length=255, nullable=true)
	 */
	public ?string $adminReply = null;

	/**
	 * @ORM\Column(name="send_user_email", type="smallint", options={"default":1})
	 */
	public int $sendUserEmail = 1;

	/**
	 * @ORM\Column(name="send_user_email_confirm", type="smallint", options={"default":0})
	 */
	public int $sendUserEmailConfirm = 0;

	/**
	 * @ORM\Column(name="send_admin_email", type="smallint", options={"default":0})
	 */
	public int $sendAdminEmail = 0;

	/**
	 * @var Submission[]|ArrayCollection
	 * @ORM\OneToMany(targetEntity="Submission", mappedBy="form")
	 */
	public $submissions;

	/**
	 * @var Field[]|ArrayCollection
	 * @ORM\OneToMany(targetEntity="Field", mappedBy="form")
	 */
	public $fields;

	/**
	 * @var FormMeasuringCodes[]|ArrayCollection
	 * @ORM\OneToMany(targetEntity="FormMeasuringCodes", mappedBy="form")
	 */
	public $measuringCodes;

	public function __construct(string $ident)
	{
		$this->setIdent($ident);
		$this->texts          = new ArrayCollection();
		$this->submissions    = new ArrayCollection();
		$this->fields         = new ArrayCollection();
		$this->measuringCodes = new ArrayCollection();
	}

	public function setIdent(string $ident)
	{
		$this->ident = str_replace(['-', ' '], ['', ''], Strings::toAscii($ident));

		return $this;
	}

	public function getIdent(): string { return $this->ident ?: ''; }

	public function setTexts(array $texts): self
	{
		$this->texts = new ArrayCollection($texts);

		return $this;
	}

	public function getText(?string $lang = null): ?FormText { return $this->texts[$lang ?: $this->locale] ?? null; }

	/** @return FormText[]|ArrayCollection */
	public function getTexts() { return $this->texts; }

	public function getUserToEmails(): array { return $this->explodeValue($this->userToEmail); }

	public function getUserToNames(): array { return $this->explodeValue($this->userToName); }

	public function getAdminToEmails(): array { return $this->explodeValue($this->adminToEmail); }

	public function getAdminToNames(): array { return $this->explodeValue($this->adminToName); }

	public function getUserBcEmails(): array { return $this->explodeValue($this->userBc); }

	public function getUserBccEmails(): array { return $this->explodeValue($this->userBcc); }

	protected function explodeValue(?string $value): array
	{
		if (!$value) {
			return [];
		}

		return array_map(static fn($v) => trim($v), explode(',', $value));
	}

	/**
	 * @return FormMeasuringCodes[]
	 */
	public function getMeasuringCodesByKey(): array
	{
		$arr = [];

		foreach ($this->measuringCodes as $ms) {
			if ($ms->locale) {
				$arr[$ms->key][$ms->locale] = $ms;
			} else {
				$arr[$ms->key] = $ms;
			}
		}

		return $arr;
	}
}
