<?php declare(strict_types = 1);

namespace Forms\AdminModule\Components;

use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use Core\Model\UI\Form\BaseForm;
use Forms\Model\DisallowedKeywords;
use Forms\Model\DisallowedKeywordsCache;
use Forms\Model\Entities\DisallowedKeyword;
use Nette\Caching\Cache;
use Nette\Utils\ArrayHash;
use Nette\Utils\Strings;
use Nette\Utils\Validators;

class DisallowedKeywordsGrid extends BaseControl
{
	public function __construct(
		protected DisallowedKeywords      $disallowedKeywords,
		protected DisallowedKeywordsCache $disallowedKeywordsCache,
	)
	{
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	public function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAjax()
			->setShowLangSwitcher(false);

		$form->addText('keyword', 'forms.disallowedKeywordsGrid.word')
			->setRequired();
		$form->addHidden('id');

		$form->addSaveCancelControl()
			->closeModalOnCancel();
		unset($form['saveControl']['save']);

		$form->onSuccess[] = function(BaseForm $form, ArrayHash $values) {
			$id = Validators::isNone($values->id) ? null : ((int) $values->id);

			if ($id) {
				$keyword = $this->disallowedKeywords->getEr()->find($id);
			} else {
				$keyword = new DisallowedKeyword($values->keyword);
			}

			$keyword->word = $values->keyword;

			$this->em->persist($keyword);
			$this->em->flush();

			$this->disallowedKeywordsCache->getCache()->clean([
				Cache::Tags => 'disallowedKeyword',
			]);

			$this['grid']->reload();
			$this->presenter->payload->hideModal = true;
		};

		return $form;
	}

	public function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();

		$dataSource = $this->disallowedKeywords->getAllWithExternals();
		uasort($dataSource, fn($arr1, $arr2) => Strings::lower($arr1['word']) <=> Strings::lower($arr2['word']));

		$grid->setDataSource($dataSource);

		$grid->addColumnText('word', 'forms.disallowedKeywordsGrid.word')
			->setFilterText();
		$grid->addAction('edit', '', 'edit!')
			->setRenderCondition(fn(array $row) => $row['id'] !== null)
			->setIcon('edit')->setBsType('primary')->addClass('ajax');
		$grid->addAction('delete', '', 'delete!')
			->setRenderCondition(fn(array $row) => $row['id'] !== null)
			->setIcon('times')->setBsType('danger')->addClass('ajax')
			->setConfirm('default.messages.reallyDelete');

		return $grid;
	}

	public function handleAdd(): void
	{
		$this->template->modalTitle = $this->t('forms.title.newDisallowedKeyword');
		$this->template->modal      = 'form';
		$this->redrawControl('modal');
	}

	public function handleEdit(int $id): void
	{
		$kw = $this->disallowedKeywords->get($id);

		$this->template->modalTitle = $this->t('forms.title.editDisallowedKeyword');
		$this->template->modal      = 'form';

		$this['form']->setDefaults([
			'id'      => $id,
			'keyword' => $kw->word,
		]);

		$this->redrawControl('modal');
	}

	public function handleDelete(int $id): void
	{
		$p = $this->presenter;
		if ($this->disallowedKeywords->remove($id)) {
			$p->flashMessageSuccess('default.removed');
			$this['grid']->reload();
		} else {
			$p->flashMessageDanger('default.removeFailed');
		}

		$p->redrawControl('flashes');
	}

}
