<?php declare(strict_types = 1);

namespace Forms\FrontModule\Model;

use Contributte\Translation\Translator;
use Core\Model\Helpers\BaseEntityService;
use Core\Model\Mailing\MailBuilderFactory;
use Core\Model\Notifiers\MailNotifiers\LogNotifier;
use Core\Model\Sites;
use Exception;
use Forms\Model\Entities\Submission;
use Forms\Model\Entities\SubmissionField;
use Forms\Model\Helpers\EmailContent;
use Nette\Utils\FileSystem;
use Tracy\Debugger;

/**
 * @method Submission|null get($id)
 */
class Submissions extends BaseEntityService
{
	protected $entityClass = Submission::class;

	public function __construct(
		protected Sites              $sites,
		protected MailBuilderFactory $mailBuilder,
		protected Translator         $translator,
	)
	{
	}

	public function confirm(int $id): bool
	{
		try {
			$submission = $this->get($id);

			if (!$submission) {
				return false;
			}

			$submission->confirmed = 1;

			$this->em->persist($submission);
			$this->em->flush();

			if ($submission->form->sendUserEmailConfirm && !$submission->confirmationEmailSent) {
				$values = [];
				try {
					$this->translator->setLocale($submission->lang);
					foreach ($this->em->getRepository(SubmissionField::class)->createQueryBuilder('sf')
						         ->select('sf.name, sf.value')
						         ->where('sf.submission = :id')->setParameter('id', $id)
						         ->getQuery()->getArrayResult() as $row) {
						$values[$row['name']] = $row['value'];
					}

					$currentSite                  = $this->sites->getCurrentSite();
					$this->mailBuilder->siteIdent = $currentSite->getIdent();
					$form                         = $submission->form;
					$texts                        = $form->getText($submission->lang);

					$replaceK = array_map(static fn($val) => '{$' . $val . '}', array_keys($values));
					$replaceV = array_values($values);

					$mail    = $this->mailBuilder->create();
					$subject = $this->translator->translate(
							'formsFront.confirmationEmail.subjectPrefix',
						) . $texts->userEmailSubject;
					$mail->setSubject($subject);
					$mail->setParameters(
						[
							'subject' => $subject,
						] + $values,
					);

					$templateFile = TMP_DIR . '/forms/' . uniqid() . '_' . $form->template . '.latte';
					FileSystem::createDir(dirname($templateFile));
					$emailContent = EmailContent::replaceLinks(
						$texts->userEmailConfirm,
						$currentSite->getCurrentDomain()
							->getDomain(),
					);
					file_put_contents($templateFile, $emailContent);

					$mail->setTemplateFile($templateFile);
					$mail->setFrom($form->userFromEmail, $form->userFromName);

					$submission->confirmEmailContent = $mail->getTemplate()->renderToString();
					$this->em->persist($submission);
					$this->em->flush();

					$toNames = $form->getUserToNames();
					foreach ($form->getUserToEmails() as $i => $email) {
						$email       = str_replace($replaceK, $replaceV, (string) $email);
						$toNames[$i] = str_replace($replaceK, $replaceV, (string) $toNames[$i]);
						$mail->addTo($email, $toNames[$i] ?? null);
					}

					if ($form->userReply) {
						$mail->getMessage()->addReplyTo($form->userReply);
					}

					$mail->send();

					$submission->confirmationEmailSent = 1;
					$this->em->persist($submission);
					$this->em->flush();

					unlink($templateFile);
				} catch (Exception $e) {
					Debugger::log($values, 'forms');
					Debugger::log($e->getMessage(), 'forms');
					LogNotifier::toDevelopers(
						"User confirm email form error - " . (isset($form) ? $form->getId() : ''),
					);
				}
			}

			return true;
		} catch (Exception) {
		}

		return false;
	}
}
