<?php declare(strict_types = 1);

namespace Forms\AdminModule\Components;

use Core\Model\Entities\QueryBuilder;
use Core\Model\Helpers\Arrays;
use Core\Model\Helpers\Strings;
use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use Exception;
use Forms\AdminModule\Model\Forms;
use Forms\AdminModule\Model\Submissions;
use Forms\AdminModule\Presenters\DefaultPresenter;
use Forms\Model\Entities\Form;
use Forms\Model\Entities\Submission;
use Forms\Model\Entities\SubmissionField;
use Forms\Model\FormsConfig;
use Nette\Application\AbortException;
use Nette\Http\Url;
use Nette\Utils\Html;
use Nette\Utils\Json;
use Nette\Utils\Validators;
use Ublaboo\DataGrid\DataSource\DoctrineDataSource;

/**
 * @property DefaultPresenter $presenter
 */
class FormSubmissions extends BaseControl
{
	/** @var int @persistent */
	public int $formId;

	protected ?Form $form = null;

	protected Submissions $submissions;
	protected Forms       $forms;

	public function __construct(
		int         $formId,
		Submissions $submissions,
		Forms       $forms
	)
	{
		$this->formId      = $formId;
		$this->submissions = $submissions;
		$this->forms       = $forms;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();

		$qb = $this->submissions->getEr()->createQueryBuilder('s')
			->where('s.form = :form')
			->leftJoin('s.fields', 'sf')
			->setParameter('form', $this->formId)
			->orderBy('s.created', 'DESC');

		$grid->setDataSource($qb);

		$submissionValues = [];
		/** @var DoctrineDataSource $dataSource */
		$dataSource                 = $grid->getDataSource();
		$dataSource->onDataLoaded[] = function($items) use (&$submissionValues) {
			$ids = array_map(static fn($row) => $row->getId(), $items);

			foreach ($this->em->getRepository(SubmissionField::class)->createQueryBuilder('sf')
				         ->addSelect('IDENTITY(sf.submission) as submission')
				         ->where('sf.submission IN (:ids)')->setParameter('ids', $ids)
				         ->getQuery()->getArrayResult() as $row) {
				$submissionValues[$row['submission']][$row[0]['name']] = $row[0]['value'];
			}
		};

		$grid->addColumnNumber('id', 'forms.grid.id');
		if (FormsConfig::load('enableCompletedState')) {
			$grid->addColumnStatus('completed', 'forms.grid.isCompleted')->setAlign('center')
				->addOption(1, 'default.yes')->setIcon('check')->setClass('btn-success')->setShowTitle(false)
				->endOption()
				->addOption(0, 'default.no')->setIcon('times')->setClass('btn-danger')->setShowTitle(false)
				->endOption()
				->onChange[] = [$this, 'gridCompletedChange'];
			$grid->getColumn('completed')
				->setFilterSelect([
					''  => '',
					'1' => $this->t('default.yes'),
					'0' => $this->t('default.no'),
				], 's.completed');
		}
		if (FormsConfig::load('enableKeywordCaptcha')) {
			$grid->addColumnText('isSpam', 'forms.grid.isSpam')
				->setFitContent()
				->setRenderer(fn(Submission $row) => $row->isSpam ? $this->t('default.yes') : $this->t('default.no'));

			$grid->getColumn('isSpam')
				->setFilterSelect([
					''  => '',
					'1' => $this->t('default.yes'),
					'0' => $this->t('default.no'),
				], 's.isSpam');
		}

		$fields           = $this->submissions->getFieldKeys($this->formId);
		$form             = $this->getForm();
		$fieldsSortConfig = null;

		if ($form) {
			$fieldsSortConfig = FormsConfig::load('adminFieldsSort.' . $form->getIdent()) ?: null;

			if (!$fieldsSortConfig) {
				$fieldsSortConfig = FormsConfig::load('adminFieldsSort.' . $form->getId()) ?: null;
			}
		}

		foreach ($fieldsSortConfig ?: $fields as $name) {
			$colCaption = $this->translator->translate('admin.forms.formSubmissions.cols.' . $name);

			// pokud neexistuje preklad, nechame jako nazev klic policka
			if (Strings::contains($colCaption, $name)) {
				$colCaption = $name;
			}
			$grid->addColumnText($name, $colCaption)
				->setRenderer(function(Submission $row) use ($name, &$submissionValues) {
					$value = $submissionValues[$row->getId()][$name] ?? null;

					$createHtml = static function($v): Html {
						$wrap = Html::el('');

						if (Validators::isUrl($v) || Strings::startsWith($v, '//')) {
							$el = Html::el('div')
								->addHtml(Html::el('a', [
									'href'   => $v,
									'target' => '_blank',
								])
									->setText(basename($v)));
						} else {
							$el = Html::el('div')
								->setText($v);
						}
						$wrap->addHtml($el);

						return $wrap;
					};

					if (Arrays::isJson($value) && !is_numeric($value)) {
						$wrap = Html::el('');

						foreach (Json::decode($value) as $v) {
							$wrap->addHtml($createHtml($v));
						}

						return $wrap;
					} else {
						$value = $createHtml((string) $value);
					}

					return $value;
				})->setFilterText()->setCondition(function(QueryBuilder $qb, $value) use ($name) {
					$qb->andWhere("sf.name = :{$name}Key AND sf.value LIKE :{$name}Val")
						->setParameter($name . 'Key', $name)
						->setParameter($name . 'Val', "%{$value}%");
				});
		}

		$grid->addColumnText('confirmed', 'forms.submission.confirmed')->setRenderer(function(Submission $row) {
			return $row->confirmed ? '1' : '0';
		})->setFilterText();
		$grid->addColumnDateTime('created', 'forms.submission.created')->setFormat('j. n. Y H:i:s');

		if (FormsConfig::load('showSendFrom')) {
			$grid->addColumnText('sendFrom', 'forms.submission.sendFrom')->setRenderer(function(Submission $row) {
				if ($row->url) {
					$url   = new Url($row->url);
					$query = $url->getQuery();

					return Html::el('a', ['href' => $row->url, 'target' => '_blank'])
						->setText($url->getPath() . ($query ? '?' . $query : ''));
				}

				return Html::el();
			})->setFilterText();
		}

		$grid->addAction('delete', '', 'delete!')
			->setIcon('times')
			->setBsType('danger')
			->addClass('ajax')
			->setConfirm('default.reallyDelete');

		$grid->getColumn('id')->getElementPrototype('td')->addClass('w1nw');
		$grid->getColumn('confirmed')->getElementPrototype('th')->addClass('w1nw');

		return $grid;
	}

	public function handleDelete(int $id): void
	{
		if ($this->submissions->remove($id)) {
			$this->presenter->flashMessageSuccess('default.removed');
		} else {
			$this->presenter->flashMessageDanger('default.removeFailed');
		}

		$this['grid']->reload();
		$this->presenter->redrawControl('flashes');
	}

	public function getForm(): ?Form
	{
		if ($this->form === null && $this->formId) {
			$this->form = $this->forms->get($this->formId);
		}

		return $this->form;
	}

	/**
	 * @param int|string $id
	 * @param int|string $newStatus
	 *
	 * @throws AbortException
	 */
	public function gridCompletedChange($id, $newStatus): void
	{
		try {
			$submission = $this->submissions->get((int) $id);

			if ($submission) {
				$submission->completed = (int) $newStatus;
				$this->em->persist($submission)->flush();
				$this->presenter->flashMessageSuccess('default.saved');
			}
		} catch (Exception $ex) {
			$this->presenter->flashMessageDanger('default.publishChangeFailed');
		}
		if ($this->presenter->isAjax()) {
			$this['grid']->redrawItem($id);
			$this->presenter->redrawControl('flashes');
		} else {
			$this->presenter->redirect('this');
		}
	}

}
