<?php declare(strict_types = 1);

namespace Forms\CronModule\Presenters;

use Core\Model\Entities\EntityManagerDecorator;
use Forms\Model\Entities\Form;
use Forms\Model\Entities\Submission;
use Forms\Model\Entities\SubmissionField;
use Forms\Model\FormsConfig;
use Nette\Security\Passwords;
use Nette\Utils\DateTime;

class ExportPresenter extends BasePresenter
{
	protected EntityManagerDecorator $em;

	public function __construct(
		EntityManagerDecorator $em
	)
	{
		$this->em = $em;
	}

	public function actionFull(
		string  $key,
		string  $siteIdent,
		string  $from,
		?string $to = null
	): void
	{
		if ($from === 'prevmonth') {
			$fromDate = (new DateTime())->modify('first day of previous month')->setTime(0, 0, 0);
			$toDate   = (new DateTime())->modify('last day of previous month')->setTime(23, 59, 59);
		} else {
			$fromDate = DateTime::createFromFormat('Y-m-d', $from)->setTime(0, 0, 0);
			$toDate   = $to
				? DateTime::createFromFormat('Y-m-d', $to)->setTime(23, 59, 59)
				: (new DateTime())->setTime(23, 59, 59);
		}

		$password  = new Passwords();
		$keyVerify = $password->verify(FormsConfig::load('exportUrlHash') . $siteIdent, $key);

		if (!$keyVerify) {
			$this->error();
		}

		$result        = [];
		$resultFormMap = [];
		$fieldsKeys    = [];
		foreach ($this->em->getRepository(Submission::class)->createQueryBuilder('s')
			         ->select('IDENTITY(s.form) as formId, s.lang, s.created, s.id')
			         ->innerJoin('s.fields', 'sf')
			         ->andWhere('s.isSpam = :isSpam')
			         ->andWhere('s.created >= :from AND s.created <= :to')
			         ->setParameters([
				         'from'   => $fromDate,
				         'to'     => $toDate,
				         'isSpam' => 0,
			         ])
			         ->orderBy('s.id', 'DESC')
			         ->getQuery()->getArrayResult() as $row) {
			$result[$row['id']]              = $row;
			$resultFormMap[$row['formId']][] = $row['id'];
		}

		if (!empty($resultFormMap)) {
			foreach ($this->em->getRepository(Form::class)->createQueryBuilder('f')
				         ->select('f.id, f.ident')
				         ->where('f.id IN (:ids)')
				         ->setParameters([
					         'ids' => array_keys($resultFormMap),
				         ])->getQuery()->getArrayResult() as $row) {

				foreach ($resultFormMap[$row['id']] as $submissionId) {
					$result[$submissionId]['formIdent'] = $row['ident'];
				}
			}

			foreach ($this->em->getRepository(SubmissionField::class)->createQueryBuilder('sf')
				         ->select('IDENTITY(sf.submission) as submissionId, sf.name, sf.value')
				         ->where('sf.submission IN (:ids)')
				         ->setParameters([
					         'ids' => array_keys($result),
				         ])->orderBy('sf.name', 'ASC')
				         ->getQuery()->getArrayResult() as $row) {
				$result[$row['submissionId']]['fields'][$row['name']] = $row['value'];
				$fieldsKeys[$row['name']]                             = $row['name'];
			}
		}

		header('Content-Encoding: UTF-8');
		header("content-type:application/csv;charset=UTF-8");
		header("Content-Disposition:attachment;filename=\"forms.csv\"");
		header('Content-Transfer-Encoding: binary');

		$fp = fopen('php://output', 'wb');
		fputs($fp, "\xEF\xBB\xBF");

		foreach ($result as $row) {
			$out = [
				$row['formIdent'],
				$row['id'],
				$row['lang'],
				$row['created'] ? $row['created']->format('Y-m-d H:i:s') : '',
			];

			foreach ($fieldsKeys as $k => $v) {
				$out[] = $k;
				$out[] = strip_tags(nl2br((string) $row['fields'][$k]));
			}

			fputcsv($fp, $out, ';');
		}

		fclose($fp);

		exit;
	}
}
