<?php declare(strict_types = 1);

namespace Forms\FrontModule\Components;

use Core\Model\Application\AppState;
use Core\Model\BotDetect;
use Core\Model\Event\ComponentTemplateEvent;
use Core\Model\Event\CreateFormEvent;
use Core\Model\Event\FormValidateEvent;
use Core\Model\Logger;
use Core\Model\Mailing\MailBuilderFactory;
use Core\Model\Parameters;
use Core\Model\Sites;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Forms\FrontModule\Model\Dao;
use Forms\FrontModule\Model\FormProcess;
use Forms\FrontModule\Model\Forms;
use Nette\Http\Request;
use Nette\Utils\ArrayHash;
use Nette\Utils\Json;

class FormControl extends BaseControl
{
	/** @var int|string */
	protected                    $formId;
	public ?string               $templateFile = null;
	protected ?Dao\Form          $form         = null;
	protected Forms              $forms;
	protected Request            $request;
	protected Sites              $sites;
	protected MailBuilderFactory $mailBuilder;
	protected FormProcess        $formProcess;

	/**
	 * @param int|string $formId
	 */
	public function __construct(
		$formId,
		Forms $forms,
		Request $request,
		Sites $sites,
		MailBuilderFactory $mailBuilder,
		FormProcess $formProcess
	)
	{
		$this->formId      = $formId;
		$this->forms       = $forms;
		$this->request     = $request;
		$this->sites       = $sites;
		$this->mailBuilder = $mailBuilder;
		$this->form        = $this->forms->getBasicData($formId);
		$this->formProcess = $formProcess;
	}

	public function render(): void
	{
		if (!$this->form) {
			return;
		}

		$this->template->thisForm = $this['form'];
		$this->template->formDao  = $this->form;

		$file = TEMPLATES_DIR . '/Front/default/Forms/FormTemplates/' . $this->form->template . '.latte';
		if (!file_exists($file)) {
			$file = PACKAGE_FORMS_DIR . '/FrontModule/Templates/FormTemplates/' . $this->form->template . '.latte';
		}

		if (!file_exists($file)) {
			foreach (Parameters::load('system.formsDir') ?: [] as $k => $v) {
				if (file_exists($v . '/' . $this->form->template . '.latte')) {
					$file = $v . '/FormTemplates/' . $this->form->template . '.latte';
					break;
				}
			}
		}

		$event                  = new ComponentTemplateEvent($this->template, $this);
		$event->data['formDao'] = $this->form;
		$this->eventDispatcher->dispatch($event, self::class . '::render');
		$this->eventDispatcher->dispatch($event, self::class . '::render' . ucfirst($this->form->template));

		$this->template->render($file);
	}

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAction($this->request->getUrl()->getAbsoluteUrl());

		$event                  = new CreateFormEvent($form, $this->getPresenterIfExists() ? $this->template : null);
		$event->control         = $this;
		$event->data['formDao'] = $this->form;
		$this->eventDispatcher->dispatch($event, self::class . '::create');
		$this->eventDispatcher->dispatch($event, self::class . '::create' . ucfirst($this->form->template));

		$form->onValidate[] = [$this, 'formValidate'];
		$form->onSuccess[]  = [$this, 'formSuccess'];

		return $form;
	}

	public function formValidate(BaseForm $form, ArrayHash $values): void
	{
		$event = new FormValidateEvent($form, $values, $this->template, $this->presenter);

		$this->eventDispatcher->dispatch($event, self::class . '::formValidate');
		$this->eventDispatcher->dispatch($event, self::class . '::formValidate' . ucfirst($this->form->template));

		if ($form->hasErrors()) {
			$this->redrawControl('formErrors');
		}
	}

	public function formSuccess(BaseForm $baseForm, ArrayHash $values): bool
	{
		if (BotDetect::isBot()) {
			$this->redrawControl('form');

			return true;
		}

		$form = $this->forms->get($this->form->ident);
		if (!$form) {
			$baseForm->addError('formsFront.sendingError');
			$this->redrawControl('formErrors');

			return false;
		}

		Logger::log('forms/send-$Y-$m-$d', AppState::getUniqueRequestId() . ' - ' . 'formSuccess - ' . Json::encode((array) $values));
		$submission = $this->formProcess->createSubmission(
			$values,
			$form,
			$baseForm,
			$this->presenter
		);

		try {
			if ($submission && !$submission->isSpam) {
				$this->formProcess->sendUser(
					$values,
					$form,
					$submission,
					$baseForm
				);

				$this->formProcess->sendAdmin(
					$values,
					$form,
					$baseForm
				);

				$this->template->formSent        = true;
				$this->template->thankYouMessage = $form->getText()->messageAfterSend;
				$this->template->formEntity      = $form;
				$this->template->submission      = $submission;

				$this->formProcess->formSuccess(
					$values,
					$form,
					$submission,
					$baseForm,
					$this->presenter
				);

				$this->redrawControl('form');
				Logger::log('forms/send-$Y-$m-$d', AppState::getUniqueRequestId() . ' - ' . 'formSuccess done');
			} else {
				Logger::log('forms/send-$Y-$m-$d', AppState::getUniqueRequestId() . ' - ' . 'formSuccess failed');
				$baseForm->addError('formsFront.sendingError');
				$this->redrawControl('formErrors');

				return false;
			}
		} catch (\Exception $e) {
			Logger::log('forms/send-$Y-$m-$d', AppState::getUniqueRequestId() . ' - ' . 'formSuccess failed ' . $e->getMessage());
			$baseForm->addError('formsFront.sendingError');
			$this->redrawControl('formErrors');

			return false;
		}

		return true;
	}

	public function getFormDao(): ?Dao\Form { return $this->form; }
}
