<?php declare(strict_types = 1);

namespace Forms\FrontModule\Model;

use Contributte\Psr7\Psr7UploadedFile;
use Contributte\Translation\Translator;
use Core\Model\Application\AppState;
use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Event\EventDispatcher;
use Core\Model\Event\FormSuccessEvent;
use Core\Model\Event\MailEvent;
use Core\Model\Helpers\Arrays;
use Core\Model\Helpers\Strings;
use Core\Model\Logger;
use Core\Model\Mailing\MailBuilderFactory;
use Core\Model\Notifiers\MailNotifiers\LogNotifier;
use Core\Model\Parameters;
use Core\Model\Sites;
use Core\Model\UI\AbstractPresenter;
use Core\Model\UI\Form\BaseForm;
use Forms\FrontModule\Components\FormControl;
use Forms\Model\DisallowedKeywords;
use Forms\Model\Entities\Form;
use Forms\Model\Entities\Submission;
use Forms\Model\Entities\SubmissionField;
use Forms\Model\FormsConfig;
use Forms\Model\Helpers\EmailContent;
use Nette\Application\LinkGenerator;
use Nette\Forms\Controls\TextArea;
use Nette\Http\FileUpload;
use Nette\Http\Request;
use Nette\Utils\ArrayHash;
use Nette\Utils\FileSystem;
use Nette\Utils\Json;
use Nette\Utils\Validators;
use Tracy\Debugger;

class FormProcess
{
	protected EventDispatcher        $eventDispatcher;
	protected Translator             $translator;
	protected Request                $request;
	protected EntityManagerDecorator $em;
	protected Sites                  $sites;
	protected MailBuilderFactory     $mailBuilder;
	protected LinkGenerator          $linkGenerator;
	protected DisallowedKeywords     $disallowedKeywords;

	public function __construct(
		EventDispatcher        $eventDispatcher,
		Translator             $translator,
		Request                $request,
		EntityManagerDecorator $em,
		Sites                  $sites,
		MailBuilderFactory     $mailBuilder,
		LinkGenerator          $linkGenerator,
		DisallowedKeywords     $disallowedKeywords
	)
	{
		$this->eventDispatcher    = $eventDispatcher;
		$this->translator         = $translator;
		$this->request            = $request;
		$this->em                 = $em;
		$this->sites              = $sites;
		$this->mailBuilder        = $mailBuilder;
		$this->linkGenerator      = $linkGenerator;
		$this->disallowedKeywords = $disallowedKeywords;
	}

	public function generateFolderName(): string { return Strings::webalize(uniqid((string) time(), true)); }

	public function uploadFiles(array $files, ?string $folderName = null): array
	{
		$result     = [];
		$folderName = $folderName ?: $this->generateFolderName();
		$fileDir    = DS . 'form-uploads' . DS . $folderName . DS;
		FileSystem::createDir(UPLOADS_DIR . $fileDir);

		foreach ($files as $file) {
			if ($file instanceof Psr7UploadedFile) {
				$name     = Strings::webalize($file->getClientFilename(), '.', false);
				$name     = str_replace(['-.', '.-'], '.', $name);
				$filePath = $fileDir . $name;

				$file->moveTo(UPLOADS_DIR . $filePath);
				$result[] = 'https://' . $this->sites->getCurrentSite()->getCurrentDomain()->getDomain() . UPLOADS_PATH . $filePath;
			} else if ($file instanceof FileUpload) {
				if ($file->isOk()) {
					$filePath = $fileDir . $file->getSanitizedName();

					$file->move(UPLOADS_DIR . $filePath);
					$result[] = 'https://' . $this->sites->getCurrentSite()->getCurrentDomain()->getDomain() . UPLOADS_PATH . $filePath;
				}
			}
		}

		return $result;
	}

	public function createSubmission(
		ArrayHash          $values,
		Form               $form,
		?BaseForm          $baseForm = null,
		?AbstractPresenter $presenter = null
	): ?Submission
	{
		$enableKeywordCaptcha = FormsConfig::load('enableKeywordCaptcha');
		$disallowedKeywords   = $enableKeywordCaptcha ? array_map(static fn(array $arr) => $arr['word'], $this->disallowedKeywords->getAllWithExternals()) : [];
		$locale               = $this->translator->getLocale();
		$texts                = $form->getText($locale);

		if (!$texts) {
			return null;
		}

		Logger::log('forms/send-$Y-$m-$d', AppState::getUniqueRequestId() . ' - ' . 'create submission');
		$event                   = new FormSuccessEvent($baseForm, $values, null, $presenter);
		$event->custom['entity'] = &$form;
		$event->custom['texts']  = &$texts;
		$this->eventDispatcher->dispatch($event, FormControl::class . '::formSuccessStart' . ucfirst($form->template));

		$submission           = new Submission($form);
		$submission->subject  = $texts->userEmailSubject;
		$submission->lang     = $locale;
		$submission->url      = $this->request->getUrl()->getAbsoluteUrl();
		$submission->userData = Json::encode([
			'REMOTE_ADDR'     => $_SERVER['REMOTE_ADDR'],
			'HTTP_USER_AGENT' => $_SERVER['HTTP_USER_AGENT'] ?? null,
			'HTTP_X_CLIENT'   => $_SERVER['HTTP_X_CLIENT'] ?? null,
		]);
		$this->em->persist($submission);

		foreach ((array) $values as $k => $v) {
			if (is_string($v)) {

				// porovnava bud slovni spojeni nebo klicova slova
				foreach ($disallowedKeywords as $disallowedKeyword) {
					$val               = mb_strtolower($v);
					$disallowedKeyword = mb_strtolower($disallowedKeyword);
					$words             = array_map('trim', Strings::split($val, '/\s+/'));

					// porovnava zakazana slovni spojeni
					if (Strings::contains($disallowedKeyword, ' ') && Strings::contains($val, $disallowedKeyword)) {
						$submission->isSpam = 1;
					}

					// vyhledava zakazana klicova slova
					foreach ($words as $w) {
						if ($w === $disallowedKeyword) {
							$submission->isSpam = 1;
						}
					}
				}

				// u textarea poli nepovolit vyplneny pouze odkaz nebo email
				if ($enableKeywordCaptcha && $baseForm && $baseForm->getComponent($k, false)
					&& is_a($baseForm->getComponent($k), TextArea::class)
					&& (Validators::isEmail($v) || Validators::isUrl($v))
				) {
					$submission->isSpam = 1;
				}
			}

			if (is_array($v)) {
				$v = Json::encode($v);
			}

			$field = new SubmissionField($submission, $k, $this->valueToString($v));
			$submission->fields->add($field);
			$this->em->persist($field);
		}

		$cookieBar = $this->request->getCookie('cookiesBar');
		$analytical = false;
		if ($cookieBar && Arrays::isJson($cookieBar)) {
			foreach (Json::decode($cookieBar, Json::FORCE_ARRAY) as $v) {
				if ($v['type'] === 'analytical') {
					$analytical = true;
					break;
				}
			}
		}

		$field = new SubmissionField($submission, 'cookieBarAnalytical', $analytical ? '1' : '0');
		$submission->fields->add($field);
		$this->em->persist($field);

		$event                   = new FormSuccessEvent($baseForm, $values, null, $presenter);
		$event->custom['entity'] = &$submission;
		$this->eventDispatcher->dispatch($event, FormControl::class . '::formSuccessBeforeSaveSubmission' . ucfirst($form->template));
		$this->em->flush();
		Logger::log('forms/send-$Y-$m-$d', AppState::getUniqueRequestId() . ' - ' . 'submission created ' . $submission->getId());

		return $submission;
	}

	public function prepareTemplateVars(ArrayHash $values): array
	{
		$currentSite = $this->sites->getCurrentSite();

		return [
			'k' => array_map(
				static fn($val) => '{$' . $val . '}',
				array_merge(
					array_keys((array) $values), ['siteName', 'siteEmail']
				)
			),
			'v' => array_merge(
				array_values(
					array_map(fn($val) => $this->valueToString($val), (array) $values)
				), [
				$currentSite->getSiteName(),
				$currentSite->getEmail(),
			]),
		];
	}

	public function sendUser(
		ArrayHash  $values,
		Form       $form,
		Submission $submission,
		?BaseForm  $baseForm = null
	): bool
	{
		Logger::log('forms/send-$Y-$m-$d', AppState::getUniqueRequestId() . ' - ' . 'start send user');
		$locale      = $this->translator->getLocale();
		$currentSite = $this->sites->getCurrentSite();
		$texts       = $form->getText($locale);

		if (!$currentSite || !$form->sendUserEmail || !$texts) {
			return false;
		}

		$this->mailBuilder->siteIdent = $currentSite->getIdent();

		$preparedTemplateVars = $this->prepareTemplateVars($values);
		$replaceK             = $preparedTemplateVars['k'] ?? [];
		$replaceV             = $preparedTemplateVars['v'] ?? [];

		try {
			$mail    = $this->mailBuilder->create();
			$subject = str_replace($replaceK, $replaceV, (string) $texts->userEmailSubject);
			$params  = [
				'subject'     => $subject,
				'confirmLink' => $form->sendUserEmailConfirm ? $this->linkGenerator->link('Forms:Front:Default:confirm', ['id' => $submission->getId()]) : null,
			];

			if ($baseForm) {
				$params += (array) $baseForm->getCustomData('extraValues', []);
			}

			$params += (array) $values;

			$mail->setSubject($subject);
			$mail->setParameters($params);

			$templateFile = TMP_DIR . '/forms/' . uniqid('', true) . '_' . $form->template . '.latte';
			FileSystem::createDir(dirname($templateFile));

			$emailContent = EmailContent::replaceLinks($texts->userEmailContent, $currentSite->getCurrentDomain()->getDomain());
			$emailContent = htmlspecialchars_decode($emailContent, ENT_QUOTES);
			file_put_contents($templateFile, $emailContent);

			$mail->setTemplateFile($templateFile);
			$mail->setFrom(
				str_replace($replaceK, $replaceV, $form->userFromEmail),
				str_replace($replaceK, $replaceV, $form->userFromName)
			);

			$toNames = $form->getUserToNames();
			foreach ($form->getUserToEmails() as $i => $email) {
				$email       = str_replace($replaceK, $replaceV, $email);
				$toNames[$i] = str_replace($replaceK, $replaceV, $toNames[$i] ?? '');
				$mail->addTo($email, $toNames[$i]);
			}

			foreach ($form->getUserBcEmails() as $email) {
				$email = str_replace($replaceK, $replaceV, $email);
				$mail->addCc($email);
			}

			foreach ($form->getUserBccEmails() as $email) {
				$email = str_replace($replaceK, $replaceV, $email);
				$mail->addBcc($email);
			}

			if ($form->userReply) {
				$mail->getMessage()->addReplyTo(str_replace($replaceK, $replaceV, $form->userReply));
			}

			$this->eventDispatcher->dispatch(new MailEvent($mail, ['values' => (array) $values]), FormControl::class . '::beforeUserEmail' . ucfirst($form->template));

			$submission->emailContent = $mail->getTemplate()->renderToString();
			$this->em->persist($submission);
			$this->em->flush();

			$mail->send();
			unlink($templateFile);

			Logger::log('forms/send-$Y-$m-$d', AppState::getUniqueRequestId() . ' - ' . 'user sent');

			return true;
		} catch (\Exception $e) {
			Debugger::log($values, 'forms');
			Logger::log('forms/send-$Y-$m-$d', AppState::getUniqueRequestId() . ' - ' . 'user sent failed');
			Debugger::log($e->getMessage(), 'forms');
			LogNotifier::toDevelopers("User email form error - " . $form->getId() . '. Error: ' . $e->getMessage());
		}

		return false;
	}

	public function sendAdmin(
		ArrayHash $values,
		Form      $form,
		?BaseForm $baseForm = null
	): bool
	{
		Logger::log('forms/send-$Y-$m-$d', AppState::getUniqueRequestId() . ' - ' . 'start send admin');
		$locale      = $this->translator->getLocale();
		$currentSite = $this->sites->getCurrentSite();
		$texts       = $form->getText($locale);

		if (!$currentSite || !$form->sendAdminEmail || !$texts) {
			return false;
		}

		$this->mailBuilder->siteIdent = $currentSite->getIdent();

		$preparedTemplateVars = $this->prepareTemplateVars($values);
		$replaceK             = $preparedTemplateVars['k'] ?? [];
		$replaceV             = $preparedTemplateVars['v'] ?? [];

		try {
			$mail    = $this->mailBuilder->create();
			$subject = str_replace($replaceK, $replaceV, (string) $texts->adminEmailSubject);
			$params  = [
				'subject' => $subject,
			];

			if ($baseForm) {
				$params += (array) $baseForm->getCustomData('extraValues', []);
			}

			$params += (array) $values;

			$mail->setSubject($subject);
			$mail->setParameters($params);

			$templateFile = TMP_DIR . '/forms/' . uniqid('', true) . '_' . $form->template . '.latte';
			FileSystem::createDir(dirname($templateFile));

			$emailContent = EmailContent::replaceLinks($texts->adminEmailContent, $currentSite->getCurrentDomain()->getDomain());
			$emailContent = htmlspecialchars_decode($emailContent, ENT_QUOTES);
			file_put_contents($templateFile, $emailContent);

			$mail->setTemplateFile($templateFile);
			$mail->setFrom(
				str_replace($replaceK, $replaceV, $form->adminFromEmail),
				str_replace($replaceK, $replaceV, $form->adminFromName)
			);

			$toNames = $form->getAdminToNames();
			foreach ($form->getAdminToEmails() as $i => $email) {
				$email       = str_replace($replaceK, $replaceV, $email);
				$toNames[$i] = str_replace($replaceK, $replaceV, $toNames[$i] ?? '');
				$mail->addTo($email, $toNames[$i]);
			}

			$cmsBcc = Parameters::load('contactForm.bcc');
			if ($cmsBcc) {
				$mail->addBcc($cmsBcc, 'PSHK CMS');
			}

			if ($form->adminReply) {
				$mail->getMessage()->addReplyTo(str_replace($replaceK, $replaceV, $form->adminReply));
			}

			$this->eventDispatcher->dispatch(new MailEvent($mail, ['values' => (array) $values]), FormControl::class . '::beforeAdminEmail' . ucfirst($form->template));

			$mail->send();
			unlink($templateFile);
			Logger::log('forms/send-$Y-$m-$d', AppState::getUniqueRequestId() . ' - ' . 'admin sent');

			return true;
		} catch (\Exception $e) {
			Debugger::log($values, 'forms');
			Debugger::log($e->getMessage(), 'forms');
			LogNotifier::toDevelopers("Admin email form error - " . $form->getId() . '. Error: ' . $e->getMessage());
		}

		return false;
	}

	public function formSuccess(
		ArrayHash          $values,
		Form               $form,
		Submission         $submission,
		?BaseForm          $baseForm = null,
		?AbstractPresenter $presenter = null
	): void
	{
		$event = new FormSuccessEvent($baseForm, $values, null, $presenter);

		$event->custom['entity']     = $form;
		$event->custom['submission'] = $submission;
		$this->eventDispatcher->dispatch($event, FormControl::class . '::formSuccess');
		$this->eventDispatcher->dispatch($event, FormControl::class . '::formSuccess' . ucfirst($form->template));
	}

	/**
	 * If the variable cannot be converted to a string, it is returned
	 *
	 * @param mixed $value
	 *
	 * @return mixed
	 */
	protected function valueToString($value)
	{
		if ($value instanceof \DateTimeImmutable) {
			return $value->format('d.m.Y H:i');
		} else if ($value instanceof FileUpload && !$value->hasFile()) {
			return '';
		}

		if (
			(!is_array($value)) &&
			((!is_object($value) && settype($value, 'string') !== false) ||
				(is_object($value) && method_exists($value, '__toString')))
		) {
			return (string) $value;
		}

		return $value;
	}
}
