<?php declare(strict_types = 1);

namespace Forms\Model\Entities;

use Core\Model\Entities\TId;
use Core\Model\Entities\TTranslateListener;
use Core\Model\Helpers\Strings;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;

/**
 * @ORM\Table(name="forms__form")
 * @ORM\Entity
 * @ORM\EntityListeners({"Core\Model\Entities\TranslateListener"})
 */
class Form
{
	use TId;
	use TTranslateListener;

	/**
	 * @ORM\Column(type="string", length=255, nullable=true)
	 */
	protected ?string $ident = null;

	/**
	 * @var Collection<string, FormText>
	 * @ORM\OneToMany(targetEntity="FormText", mappedBy="form", indexBy="lang")
	 */
	public Collection $texts;

	/**
	 * @ORM\Column(type="string", length=255, nullable=true)
	 */
	public ?string $template = null;

	/**
	 * @ORM\Column(type="smallint", options={"default":1})
	 */
	public int $isPublished = 1;

	/**
	 * @ORM\Column(type="string", length=255, nullable=true)
	 */
	public ?string $userFromEmail = null;

	/**
	 * @ORM\Column(type="string", length=255, nullable=true)
	 */
	public ?string $userFromName = null;

	/**
	 * @ORM\Column(type="string", length=255, nullable=true)
	 */
	public ?string $userToEmail = null;

	/**
	 * @ORM\Column(type="string", length=255, nullable=true)
	 */
	public ?string $userToName = null;

	/**
	 * @ORM\Column(type="string", length=255, nullable=true)
	 */
	public ?string $userBc = null;

	/**
	 * @ORM\Column(type="string", length=255, nullable=true)
	 */
	public ?string $userBcc = null;

	/**
	 * @ORM\Column(type="string", length=255, nullable=true)
	 */
	public ?string $userReply = null;

	/**
	 * @ORM\Column(name="admin_from_email", type="string", length=255, nullable=true)
	 */
	public ?string $adminFromEmail = null;

	/**
	 * @ORM\Column(type="string", length=255, nullable=true)
	 */
	public ?string $adminFromName = null;

	/**
	 * @ORM\Column(type="string", length=255, nullable=true)
	 */
	public ?string $adminToEmail = null;

	/**
	 * @ORM\Column(type="string", length=255, nullable=true)
	 */
	public ?string $adminToName = null;

	/**
	 * @ORM\Column(type="string", length=255, nullable=true)
	 */
	public ?string $adminReply = null;

	/**
	 * @ORM\Column( type="smallint", options={"default":1})
	 */
	public int $sendUserEmail = 1;

	/**
	 * @ORM\Column(type="smallint", options={"default":0})
	 */
	public int $sendUserEmailConfirm = 0;

	/**
	 * @ORM\Column(type="smallint", options={"default":0})
	 */
	public int $sendAdminEmail = 0;

	/**
	 * @var Collection<Submission>
	 * @ORM\OneToMany(targetEntity="Submission", mappedBy="form")
	 */
	public Collection $submissions;

	/**
	 * @var Collection<Field>
	 * @ORM\OneToMany(targetEntity="Field", mappedBy="form")
	 */
	public Collection $fields;

	/**
	 * @var Collection<FormMeasuringCodes>
	 * @ORM\OneToMany(targetEntity="FormMeasuringCodes", mappedBy="form")
	 */
	public Collection $measuringCodes;

	/**
	 * @ORM\Column(type="array", nullable=true)
	 */
	public ?array $settings = null;

	public function __construct(string $ident)
	{
		$this->setIdent($ident);
		$this->texts          = new ArrayCollection;
		$this->submissions    = new ArrayCollection;
		$this->fields         = new ArrayCollection;
		$this->measuringCodes = new ArrayCollection;
	}

	public function setIdent(string $ident): self
	{
		$this->ident = str_replace(['-', ' '], ['', ''], Strings::toAscii($ident));

		return $this;
	}

	public function getIdent(): string { return $this->ident ?: ''; }

	public function setTexts(array $texts): self
	{
		$this->texts = new ArrayCollection($texts);

		return $this;
	}

	public function getText(?string $lang = null): ?FormText { return $this->texts[$lang ?: $this->locale] ?? null; }

	/** @return Collection<FormText> */
	public function getTexts(): Collection { return $this->texts; }

	public function getUserToEmails(): array { return $this->explodeValue($this->userToEmail); }

	public function getUserToNames(): array { return $this->explodeValue($this->userToName); }

	public function getAdminToEmails(): array { return $this->explodeValue($this->adminToEmail); }

	public function getAdminToNames(): array { return $this->explodeValue($this->adminToName); }

	public function getUserBcEmails(): array { return $this->explodeValue($this->userBc); }

	public function getUserBccEmails(): array { return $this->explodeValue($this->userBcc); }

	protected function explodeValue(?string $value): array
	{
		if (!$value) {
			return [];
		}

		return array_map(static fn($v) => trim($v), explode(',', $value));
	}

	/**
	 * @return FormMeasuringCodes[]
	 */
	public function getMeasuringCodesByKey(): array
	{
		$arr = [];

		foreach ($this->measuringCodes as $ms) {
			if ($ms->locale) {
				$arr[$ms->key][$ms->locale] = $ms;
			} else {
				$arr[$ms->key] = $ms;
			}
		}

		return $arr;
	}

}
