<?php declare(strict_types = 1);

namespace Forms\AdminModule\Components;

use Core\Model\Helpers\Arrays;
use Core\Model\Helpers\FormHelper;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Core\Model\UI\Form\Controls\EditorInput;
use Exception;
use Forms\AdminModule\Model\Forms;
use Forms\FrontModule\Components\IFormControlFactory;
use Forms\Model\Entities\Form;
use Forms\Model\Entities\FormMeasuringCodes;
use Forms\Model\Entities\FormText;
use Forms\Model\FormsCache;
use Forms\Model\FormsConfig;
use MeasuringCodes\DI\MeasuringCodesExtension;
use Nette\Forms\Controls\Button;
use Nette\Forms\Controls\TextArea;
use Nette\Forms\Controls\UploadControl;
use Nette\InvalidArgumentException;
use Nette\Utils\ArrayHash;

class FormForm extends BaseControl
{
	/** @var int|null @persistent */
	public ?int $formId = null;

	protected ?Form $form = null;

	protected ?BaseForm $cFrontForm       = null;
	protected ?array    $cFrontFormFields = null;

	protected array $userTextFields = ['userEmailSubject', 'userFromEmail', 'userFromName', 'userToEmail', 'userToName',
		'userBc', 'userBcc', 'userReply'];

	protected array $adminTextFields = ['adminEmailSubject', 'adminFromEmail', 'adminFromName', 'adminToEmail',
		'adminToName', 'adminReply'];

	public function __construct(
		protected Forms               $forms,
		protected FormsCache          $formsCache,
		protected IFormControlFactory $formControlFactory,
	)
	{
	}

	public function render(): void
	{
		$this->template->userTextFields  = $this->userTextFields;
		$this->template->adminTextFields = $this->adminTextFields;
		$this->template->thisForm        = $this['form'];
		$this->template->frontFormFields = $this->getFrontFormFields();

		$this->template->render($this->getTemplateFile());
	}

	protected function createFrontForm(): ?BaseForm
	{
		if ($this->cFrontForm === null) {
			$formEntity = $this->getForm();

			if (!$formEntity) {
				$this->presenter->flashMessageWarning('forms.form.notFound');
				$this->presenter->redrawControl('flashes');

				return null;
			}

			$formControl = $this->formControlFactory->create($formEntity->getId());

			$this->cFrontForm = $formControl['form'];
		}

		return $this->cFrontForm;
	}

	protected function getFrontFormFields(): array
	{
		if ($this->cFrontFormFields === null) {
			$form = $this->createFrontForm();
			if (!$form) {
				return [];
			}

			$this->cFrontFormFields = [];

			foreach ($form->getComponents(true) as $component) {
				/** @var \Nette\Forms\Controls\BaseControl $component */
				if (
					$component instanceof Button
					|| !method_exists($component, 'getCaption')
					|| Arrays::contains(FormsConfig::load('adminGetFrontFormFieldsSkip'), $component->getName())
				) {
					continue;
				}

				$this->cFrontFormFields[$component->getName()] = [
					'component' => $component,
					'caption'   => $this->t($component->getCaption()),
				];
			}
		}

		return $this->cFrontFormFields;
	}

	public function getFieldsDefaults(): array
	{
		$form = $this->createFrontForm();

		if (!$form) {
			return [];
		}

		$result = [
			'userEmailSubject'  => $this->t('forms.defaults.userEmailSubject'),
			'userFromEmail'     => $this->t('forms.defaults.userFromEmail'),
			'userFromName'      => $this->t('forms.defaults.userFromName'),
			'userToEmail'       => $this->t('forms.defaults.userToEmail'),
			'userToName'        => $this->t('forms.defaults.userToName'),
			'userReply'         => $this->t('forms.defaults.userReply'),
			'adminEmailSubject' => $this->t('forms.defaults.adminEmailSubject'),
			'adminFromEmail'    => $this->t('forms.defaults.adminFromEmail'),
			'adminFromName'     => $this->t('forms.defaults.adminFromName'),
			'adminToEmail'      => $this->t('forms.defaults.adminToEmail'),
			'adminToName'       => $this->t('forms.defaults.adminToName'),
			'adminReply'        => $this->t('forms.defaults.adminReply'),
		];

		$message       = '';
		$messageFields = [];
		$files         = [];

		foreach ($this->getFrontFormFields() as $name => $v) {
			if ($v['component'] instanceof TextArea) {
				$messageFields[] = '<p>' . $v['caption'] . ': <br>{$' . $name . '|breaklines|noescape}</p>';
			} else if ($v['component'] instanceof UploadControl) {
				$files[] = '<div n:foreach="$' . $name . ' as $file"><a href="{$file}">{basename($file)}</a></div>';
			} else {
				$messageFields[] = $v['caption'] . ': {$' . $name . '}';
			}
		}

		if (!empty($messageFields)) {
			$message .= '<p>' . implode('<br>', $messageFields) . '</p>';
		}

		if (!empty($files)) {
			$message .= implode('<br>', $files);
		}

		$result['userEmailContent']  = '<p>Potvrzení přijetí zprávy z webu {$siteName}</p>' . $message;
		$result['adminEmailContent'] = '<p>Máte novou zprávu z webu {$siteName}</p>' . $message;

		return $result;
	}

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAjax();

		$form->addText('title', 'forms.form.title')
			->setIsMultilanguage();
		$form->addText('ident', 'forms.form.ident')
			->setRequired();
		$form->addText('subject', 'forms.form.subject')
			->setIsMultilanguage();
		$form->addEditor('messageAfterSend', 'forms.form.messageAfterSend')
			->setDisableAutoP(false)
			->setHeight(100)
			->setIsMultilanguage();
		$form->addBool('isPublished', 'default.isPublished')
			->setDefaultValue(1);

		foreach (array_merge($this->userTextFields, $this->adminTextFields) as $col) {
			$form->addText($col, 'forms.form.' . $col)
				->setHtmlAttribute('data-default',);
		}

		$form->getComponent('userEmailSubject')->setIsMultiLanguage();
		$form->getComponent('adminEmailSubject')->setIsMultiLanguage();

		$templates = [];
		foreach (glob(PACKAGE_FORMS_DIR . '/FrontModule/Templates/EmailTemplates/*.latte') as $file) {
			$fileName             = pathinfo((string) $file, PATHINFO_FILENAME);
			$templates[$fileName] = 'forms.templates.' . $fileName;
		}
		foreach (glob(TEMPLATES_DIR . '/Front/default/Forms/EmailTemplates/*.latte') as $file) {
			$fileName             = pathinfo((string) $file, PATHINFO_FILENAME);
			$templates[$fileName] = 'forms.templates.' . $fileName;
		}
		$form->addSelect('template', 'forms.form.template', $templates);

		$form->addCheckbox('sendUserEmail', 'forms.form.sendUserEmail')
			->setDefaultValue(1);
		$form->addCheckbox('sendUserEmailConfirm', 'forms.form.sendUserEmailConfirm')
			->setDefaultValue(1);
		$form->addCheckbox('sendAdminEmail', 'forms.form.sendAdminEmail')
			->setDefaultValue(0);
		$form->addEditor('userEmailContent', 'forms.form.userEmailContent')
			->setIsMultilanguage();
		$form->addEditor('userEmailConfirm', 'forms.form.userEmailConfirm')
			->setIsMultilanguage();
		$form->addEditor('adminEmailContent', 'forms.form.adminEmailContent')
			->setIsMultilanguage();

		if (class_exists(MeasuringCodesExtension::class)) {
			$container = $form->addContainer('measuringCodes');
			$container->addText('googleAdsConversionLabel', 'forms.measuringCodes.googleAdsConversionLabel');
			$container->addText('sklikConversionId', 'forms.measuringCodes.sklikConversionId');
		}

		$form->addSaveCancelControl();

		$form->onValidate[] = $this->formValidate(...);
		$form->onSuccess[]  = $this->formSuccess(...);

		return $form;
	}

	public function formValidate(BaseForm $form, ArrayHash $values): bool
	{
		if (!$this->forms->validateFormIdent($values->ident, $this->formId)) {
			$form->addError('forms.form.identExist');
			$this->redrawControl('form');

			return false;
		}

		return true;
	}

	public function formSuccess(BaseForm $baseForm, ArrayHash $values): bool
	{
		try {
			$langValues = $baseForm->convertMultilangValuesToArray();

			if ($this->formId) {
				$form  = $this->forms->get($this->formId);
				$texts = $form->getTexts()->toArray();
			} else {
				$form  = new Form($values->ident);
				$texts = [];
			}

			foreach ($langValues as $l => $v) {
				if (!isset($texts[$l])) {
					$texts[$l] = new FormText($form, $l, $v['title']);
				}

				$texts[$l]->title            = $v['title'];
				$texts[$l]->messageAfterSend = $v['messageAfterSend'] ?: null;

				foreach (['userEmailSubject', 'userEmailContent', 'adminEmailSubject', 'adminEmailContent'] as $col) {
					$texts[$l]->$col = $v[$col];
					unset($values->$col);
				}

				$texts[$l]->userEmailConfirm = $v['userEmailConfirm'];
				unset($values->userEmailConfirm);

				$this->em->persist($texts[$l]);
				$form->texts->set($l, $texts[$l]);
			}

			foreach (['sendUserEmail', 'sendUserEmailConfirm', 'sendAdminEmail'] as $col) {
				$form->$col = (int) $values->$col;
				unset($values->$col);
			}

			$form->setIdent($values->ident);
			$this->em->persist($form);

			// TODO multijazykove
			if (class_exists(MeasuringCodesExtension::class)) {
				$currentMS = $form->getMeasuringCodesByKey();

				foreach ($values->measuringCodes as $k => $v) {
					if (!isset($currentMS[$k])) {
						$ms = new FormMeasuringCodes($form, $k);
						$form->measuringCodes->add($ms);
					} else {
						$ms = $currentMS[$k];
					}

					$ms->value = $v;

					$this->em->persist($ms);
				}

				// delete
				foreach (array_diff_key($currentMS, (array) $values->measuringCodes) as $v) {
					$this->em->remove($v);
				}

				unset($values->measuringCodes);
			}

			FormHelper::fillEntityByValues($form, $values);
			$this->em->persist($form);

			$this->em->flush();

			$this->presenter->flashMessageSuccess('default.saved');
			$this->presenter->redrawControl('flashes');

			$baseForm->addCustomData('formId', $form->getId());

			$this->formsCache->getCache()->remove('form_' . $form->getId());
			$this->formsCache->getCache()->remove('form_' . $form->getIdent());

			return true;
		} catch (Exception) {
			$this->presenter->flashMessageDanger('default.error');
			$this->presenter->redrawControl('flashes');
		}

		return false;
	}

	public function setForm(int $formId): void
	{
		$this->formId = $formId;
		$form         = $this->forms->get($formId);

		if (!$form) {
			throw new InvalidArgumentException;
		}

		FormHelper::fillContainerByEntity($this['form'], $form);

		if (class_exists(MeasuringCodesExtension::class)) {
			$ms = $form->getMeasuringCodesByKey();
			foreach ($this['form']->getComponent('measuringCodes')->getComponents() as $c) {
				$c->setDefaultValue($ms[$c->getName()] ? $ms[$c->getName()]->value : null);
			}
		}

		$texts = [
			'ident' => $form->getIdent(),
		];
		foreach ($form->getTexts()->toArray() as $l => $e) {
			/** @var FormText $e */
			$texts['title'][$l]            = $e->title;
			$texts['messageAfterSend'][$l] = $e->messageAfterSend;
			foreach (['userEmailSubject', 'userEmailContent', 'userEmailConfirm', 'adminEmailSubject',
				         'adminEmailContent'] as $col) {
				$texts[$col][$l] = $e->$col;
			}
		}

		$this['form']->setDefaults($texts);

		foreach ($this->getFieldsDefaults() as $k => $v) {
			/** @var \Nette\Forms\Controls\BaseControl|null $component */
			$component = $this['form']->getComponent($k, false);
			if ($component === null) {
				continue;
			}

			if ($component instanceof EditorInput) {
				$component->setHtmlAttribute('data-field-default-editor', $v);
			} else {
				$component->setHtmlAttribute('data-field-default', $v);
			}
		}
	}

	public function getForm(): ?Form
	{
		if ($this->form === null && $this->formId) {
			$this->form = $this->forms->get($this->formId);
		}

		return $this->form;
	}
}
