<?php declare(strict_types = 1);

namespace Forms\AdminModule\Model;

use Core\Model\Helpers\BaseEntityService;
use Doctrine\ORM\Query\Expr\Join;
use Forms\Model\Entities\Submission;
use Forms\Model\Entities\SubmissionField;
use Forms\Model\FormsConfig;

/**
 * @method Submission|object|null getReference($id)
 * @method Submission|object|null get($id)
 */
class Submissions extends BaseEntityService
{
	protected $entityClass = Submission::class;

	public function getAllCount(): array
	{
		$result = [];

		foreach ($this->getEr()->createQueryBuilder('s')
			         ->select('IDENTITY(s.form) as form, COUNT(s.id) as count')
			         ->groupBy('s.form')->getQuery()->getArrayResult() as $row) {
			$result[$row['form']] = $row['count'];
		}

		return $result;
	}

	public function getFieldKeys(int $formId): array
	{
		$arr = [];

		foreach ($this->em->getRepository(SubmissionField::class)->createQueryBuilder('sf')
			         ->select('sf.name')
			         ->innerJoin('sf.submission', 's', Join::WITH, 's.form = :form')
			         ->setParameter('form', $formId)
			         ->groupBy('sf.name')->getQuery()->getArrayResult() as $row) {
			$arr[] = $row['name'];
		}

		return $arr;
	}

	public function exportCsv(int $formId): string
	{
		$submissionValues = [];

		foreach ($this->em->getRepository(SubmissionField::class)->createQueryBuilder('sf')
			         ->addSelect('IDENTITY(sf.submission) as submission')
			         ->innerJoin('sf.submission', 's', Join::WITH, 's.form = :form')
			         ->setParameter('form', $formId)
			         ->getQuery()->getArrayResult() as $row) {
			$submissionValues[$row['submission']][$row[0]['name']] = $row[0]['value'];
		}

		foreach ($this->getEr()->createQueryBuilder('s')
			         ->select('s.id, s.confirmed, s.created, s.completed')
			         ->where('s.form = :form')
			         ->setParameter('form', $formId)
			         ->getQuery()->getArrayResult() as $row) {
			if (FormsConfig::load('enableCompletedState')) {
				$submissionValues[$row['id']]['completed'] = $row['completed'];
			}
			$submissionValues[$row['id']]['confirmed'] = $row['confirmed'];
			$submissionValues[$row['id']]['created']   = $row['created']->format('j. n. Y H:i:s');
		}

		$fieldKeys   = $this->getFieldKeys($formId);
		$fieldKeys[] = 'completed';
		$fieldKeys[] = 'confirmed';
		$fieldKeys[] = 'created';
		$csv         = '"' . implode('";"', $fieldKeys) . '"' . PHP_EOL;
		foreach ($submissionValues as $submission => $vals) {
			$row = [];
			foreach ($fieldKeys as $fieldKey) {
				$row[] = $vals[$fieldKey] ?? '';
			}

			$csv .= '"' . implode('";"', $row) . '"' . PHP_EOL;
		}

		return $csv;
	}
}
