<?php declare(strict_types = 1);

namespace Gallery\AdminModule\Components\Album;

use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Gallery\AdminModule\Components\Image\IImagesZoneFactory;
use Gallery\AdminModule\Components\Image\ImagesZone;
use Gallery\Model\Albums;
use Gallery\Model\Entities\Album;
use Nette\Http\IResponse;
use Nette\Utils\ArrayHash;
use Nette\Utils\DateTime;

class AlbumForm extends BaseControl
{
	/** @var IImagesZoneFactory */
	protected $imagesZoneFactory;

	/** @var Album */
	protected $album;

	/** @var Albums */
	protected $albumsService;

	private $albumBasePath = UPLOADS_PATH . '/gallery';

	public function __construct(IImagesZoneFactory $imagesZoneFactory, Albums $albums)
	{
		$this->imagesZoneFactory = $imagesZoneFactory;
		$this->albumsService     = $albums;
	}

	public function render()
	{
		$this->template->render(__DIR__ . '/AlbumForm.latte');
	}

	protected function createComponentForm()
	{
		$form = $this->createForm();

		$form->addText('title', 'gallery.albumForm.title')->setMaxLength(255);
		$form->addTextArea('description', 'gallery.albumForm.description');
		$form->addBool('isPublished', 'gallery.albumForm.isPublished')->setDefaultValue(1);

		$form->addHidden('preparedAlbumId');
		$form->addSubmit('submit', 'default.save');

		if ($this->album) {
			$a = $this->album;
			$form->setDefaults([
				'title'       => $a->title,
				'description' => $a->description,
			]);
		}

		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values)
	{
		try {
			$presenter = $this->getPresenter();
			$user      = $presenter->getUser();

			if ($this->album)
				$album = $this->album;
			else if ($values->preparedAlbumId)
				$album = $this->albumsService->getAlbum($values->preparedAlbumId);
			else
				$album = new Album($this->albumBasePath);

			$album->title       = $values->title != '' ? $values->title : $this->t('gallery.album.withoutName') . ' - ' . (new DateTime())->format('j. n. Y');
			$album->description = $values->description;
			$album->isPublished = $values->isPublished;

			$this->em->persist($album);
			$this->em->flush();
		} catch (\Exception $e) {
			$form->addError($e->getMessage());

			return false;
		}
	}

	/*******************************************************************************************************************
	 * =================  Get/Set
	 */

	public function setAlbum($id)
	{
		$this->album = $this->em->getRepository(Album::class)->find($id);

		if (!$this->album)
			$this->getPresenter()->error(null, IResponse::S404_NOT_FOUND);
	}

	/*******************************************************************************************************************
	 * ============================== Components
	 */

	protected function createComponentImagesZone()
	{
		$control = $this->imagesZoneFactory->create();

		if ($this->album) {
			$control->setAlbum($this->album->getId());
		}

		$control->onEmpty[] = function($control) {
			/** @var ImagesZone $control */

			if (!$this->album) {
				$album        = new Album($this->albumBasePath);
				$album->title = $this->t('gallery.album.withoutName') . ' - ' . (new DateTime())->format('j. n. Y');
				$this->em->persist($album);
				$this->em->flush();

				$control->setAlbum($album->getId());

				$control->extendedResponseOnEmpty['albumId'] = $album->getId();
			}
		};

		return $control;
	}
}
