<?php declare(strict_types = 1);

namespace Gallery\Model\Entities;

use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Criteria;
use Doctrine\ORM\Mapping as ORM;
use Gedmo\Mapping\Annotation as Gedmo;
use Kdyby\Doctrine\Entities\Attributes\Identifier;
use Kdyby\Doctrine\Entities\MagicAccessors;
use Nette\Utils\Strings;

/**
 * @ORM\Table(name="gallery__album")
 * @ORM\Entity
 * @ORM\EntityListeners({"AlbumListener"})
 */
class Album
{
	use Identifier;
	use MagicAccessors;

	/**
	 * @var string
	 * @ORM\Column(name="title", type="string", length=255, nullable=true)
	 */
	public $title;

	/**
	 * @var string
	 * @ORM\Column(name="description", type="text", nullable=true)
	 */
	public $description;

	/**
	 * @var string
	 * @ORM\Column(name="base_path", type="string", nullable=false)
	 */
	public $basePath;

	/**
	 * @var Image[]
	 * @ORM\OneToMany(targetEntity="Image", mappedBy="album")
	 */
	protected $images;

	protected $cover;

	/**
	 * @var int
	 * @ORM\Column(name="is_published", type="smallint")
	 */
	public $isPublished;

	public function __construct($basePath)
	{
		$this->basePath    = $basePath;
		$this->isPublished = 1;
		$this->images      = new ArrayCollection();
	}

	public function setId($id)
	{
		$this->id = $id;
	}

	public function generatePath() { return $this->basePath . '/' . $this->getId() . '-' . Strings::webalize($this->title); }

	public function getPath() { return $this->basePath . '/album_' . $this->getId(); }

	public function getImages()
	{
		$iterator = $this->images->getIterator();
		$iterator->uasort(function($a, $b) {
			return ($a->getPosition() < $b->getPosition()) ? -1 : 1;
		});

		return new ArrayCollection(iterator_to_array($iterator));
	}

	public function setCoverImage($img)
	{
		$this->cover = $img;
	}

	public function getCoverRaw() { return $this->cover; }

	/**
	 * @return Image|null
	 */
	public function getCoverImage()
	{
		if (!$this->cover) {
			if ($this->images) {
				foreach ($this->images->toArray() as $img) {
					if ($img->isCover) {
						$cover = $img;
						break;
					}
				}

				if (!isset($cover))
					$cover = $this->images->first();

				$this->cover = $cover;
			}
		}

		return $this->cover;
	}

	/**
	 * @return int
	 */
	public function getImagesCount()
	{
		return $this->images->count();
	}
}