<?php declare(strict_types = 1);

namespace Gallery\Model\Entities;

use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Mapping as ORM;
use Gedmo\Mapping\Annotation as Gedmo;
use Kdyby\Doctrine\Entities\Attributes\Identifier;
use Kdyby\Doctrine\Entities\MagicAccessors;
use Nette\Utils\DateTime;
use Nette\Utils\Json;
use Tags\Model\Entities\Tag;

/**
 * @ORM\Table(name="gallery__image")
 * @ORM\Entity
 */
class Image
{
	use Identifier;
	use MagicAccessors;

	/**
	 * @var string
	 * @ORM\Column(name="source", type="string", length=255, nullable=true)
	 */
	public $source;

	/**
	 * @var string
	 * @ORM\Column(name="title", type="text", nullable=true)
	 */
	public $title;

	/**
	 * @var string
	 * @ORM\Column(name="description", type="text", nullable=true)
	 */
	public $description;

	/**
	 * @var string
	 * @ORM\Column(name="link", type="text", nullable=true)
	 */
	public $link;

	/**
	 * @var string
	 * @ORM\Column(name="filename", type="string", nullable=false)
	 */
	protected $filename;

	/**
	 * @var string
	 * @ORM\Column(name="path", type="string", nullable=true)
	 */
	public $path;

	/**
	 * TODO manyToMany -> nebude potřeba cloneOf. Budou potřeba úpravy i v komponentě
	 *
	 * @var Album
	 * @Gedmo\SortableGroup
	 * @ORM\ManyToOne(targetEntity="Album", inversedBy="images")
	 * @ORM\JoinColumn(name="album_id", referencedColumnName="id", onDelete="CASCADE")
	 */
	protected $album;

	/**
	 * @var Tag[]
	 * @ORM\ManyToMany(targetEntity="Tags\Model\Entities\Tag")
	 * @ORM\JoinTable(name="gallery__image_tags",
	 *     joinColumns={@ORM\JoinColumn(name="image_id", referencedColumnName="id", onDelete="CASCADE")},
	 *     inverseJoinColumns={@ORM\JoinColumn(name="tag_id", referencedColumnName="id", onDelete="CASCADE")}
	 *     )
	 */
	protected $tags;

	/**
	 * @var int
	 *
	 * @Gedmo\SortablePosition
	 * @ORM\Column(name="position", type="integer")
	 */
	private $position;

	/**
	 * @var int
	 * @ORM\Column(name="is_published", type="smallint")
	 */
	public $isPublished;

	/**
	 * @var int
	 * @ORM\Column(name="is_cover", type="smallint")
	 */
	public $isCover;

	/**
	 * @var array
	 * @ORM\Column(name="thumb_coords", type="array", nullable=true)
	 */
	protected $thumbCoords;

	/**
	 * @var Image
	 * @ORM\ManyToOne(targetEntity="Image")
	 * @ORM\JoinColumn(name="clone_of_id", referencedColumnName="id", onDelete="SET NULL", nullable=true)
	 *
	 */
	protected $cloneOf;

	/**
	 * @var DateTime
	 * @ORM\Column(name="last_use", type="datetime", nullable=true)
	 */
	private $lastUse;

	/**
	 * Image constructor.
	 *
	 * @param Album  $album
	 * @param string $filename
	 */
	public function __construct($album, $filename)
	{
		$this->album       = $album;
		$this->filename    = $filename;
		$this->isPublished = 1;
		$this->isCover     = 0;
		$this->tags        = new ArrayCollection();
		$this->path        = $album->generatePath();
		$this->setPosition(-1);
		$this->lastUse = new DateTime();
	}

	public function clear()
	{
		$this->source      = '';
		$this->description = '';
		$this->link        = '';
		$this->isPublished = 1;
		$this->isCover     = 0;
		$this->setTags([]);
		$this->setPosition(-1);
	}

	public function getFilename() { return $this->filename; }

	public function getFilePath() { return $this->path . '/' . $this->getFilename(); }

	public function getFile() { return WWW_DIR . $this->getFilePath(); }

	public function getSize()
	{
		$file = $this->getFile();

		if (file_exists($file))
			return filesize($file);

		return 0;
	}

	/******
	 * === Album
	 */

	public function getAlbum() { return $this->album; }

	public function setAlbum($album)
	{
		$this->album   = $album;
		$this->lastUse = new DateTime();
	}

	/******
	 * === Clone of
	 */

	public function getCloneOf() { return $this->cloneOf; }

	public function setCloneOf($image)
	{
		$this->cloneOf = $image;
		$this->lastUse = new DateTime();
	}

	/******
	 * === Thumb coords
	 */

	public function getThumbCoords()
	{
		try {
			if (!is_array($this->thumbCoords))
				$this->thumbCoords = Json::decode($this->thumbCoords, Json::FORCE_ARRAY);

			return $this->thumbCoords;
		} catch (\Exception $e) {
		}

		return null;
	}

	/******
	 * === Last use
	 */

	public function setLastUse() { $this->lastUse = new DateTime(); }

	public function getLastUse() { return $this->lastUse; }

	/******
	 * === Tags
	 */

	public function addTag($tag)
	{
		if ($this->tags->contains($tag))
			return;

		$this->tags->add($tag);
	}

	public function removeTag($tag)
	{
		if (!$this->tags->contains($tag))
			return;

		$this->tags->removeElement($tag);
	}

	public function setTags(array $tags)
	{
		$this->tags = new ArrayCollection($tags);
	}

	public function getTags() { return $this->tags; }

	public function getTagsTitles()
	{
		$tags = [];
		foreach ($this->getTags() as $tag)
			$tags[$tag->getId()] = $tag->title;

		return $tags;
	}

	/******
	 * === Position
	 */

	public function setPosition($position)
	{
		$this->position = $position;
	}

	public function getPosition()
	{
		return $this->position;
	}

	public function __clone()
	{
		$this->id      = null;
		$this->lastUse = new DateTime();
	}

}