<?php declare(strict_types = 1);

namespace Gallery\FrontModule\Model;

use Core\Model\Helpers\BaseFrontEntityService;
use Core\Model\Lang\DefaultLang;
use Gallery\Model\Entities\Image;
use Gallery\FrontModule\Model\Dao;

/**
 * Class Images
 * @package Gallery\FrontModule\Model
 */
class Images extends BaseFrontEntityService
{
	const CACHE_NAMESPACE = 'images';

	protected $entityClass = Image::class;

	/** @var DefaultLang @inject */
	public $defaultLang;

	public function getCoverImage($albumId)
	{
		if (!$albumId)
			return null;

		$r = $this->getEr()->createQueryBuilder('i')->where('i.isCover = 1 OR i.position = 0')
			->join('i.album', 'a')->andWhere('a.id = :albumId')->setParameter('albumId', $albumId)
			->getQuery()->useResultCache(true, 60)->getResult();

		foreach ($r as $i) {
			if ($i->isCover)
				return $i;
		}

		return $r ? $r[0] : null;
	}

	/**
	 * @param int $albumId
	 *
	 * @return Dao\Image[]
	 */
	public function getImagesByAlbum($albumId)
	{
		$images = $this->getEr()->createQueryBuilder('i', 'i.id')
			->addSelect('it')
			->where('i.isPublished = 1')
			->andWhere('i.album = :album')->setParameter('album', $albumId)
			->leftJoin('i.texts', 'it', 'WITH', 'it.lang = :lang')
			->setParameter('lang', $this->defaultLang->locale)
			->orderBy('i.position')
			->groupBy('i.id')
			->getQuery()->getArrayResult();

		return $this->fillDao($images);
	}

	/**
	 * @param Image[] $images
	 *
	 * @return Dao\Image[]
	 */
	protected function fillDao($images)
	{
		$imgs   = [];
		$locale = $this->defaultLang->locale;

		foreach ($images as $img) {
			$i = (new Dao\Image())
				->setId($img['id'])
				->setTitle($img['texts'][$locale]['title'])
				->setDescription($img['texts'][$locale]['description'])
				->setSource($img['texts'][$locale]['source'])
				->setLink($img['texts'][$locale]['link'])
				->setFile($img['path'] . DIRECTORY_SEPARATOR . $img['filename'])
				->setPosition($img['position'])
				->setIsCover($img['isCover'])
				->setThumbCoords($img['thumbCoords']);

			$imgs[$img['id']] = $i;
		}

		return $imgs;
	}
}
