<?php declare(strict_types = 1);

namespace Gallery\AdminModule\Components\Image;

use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Lang\Lang;
use Core\Model\UI\BaseControl;
use Gallery\Model\Entities\Album;
use Gallery\Model\Entities\Image;
use Gallery\Model\Entities\ImageText;
use Gallery\Model\Images;
use Nette\Http\FileUpload;
use Nette\Http\IResponse;
use Nette\Utils\Html;
use Nette\Utils\Image as NImage;
use Nette\Utils\Json;
use Tags\Model\Tags;

/**
 * TODO popup náhled celé fotky fotky
 * TODO dodělat info o ajax stavu u každéh tlačítka
 * TODO handle requesty cele přes post (i ID) a refaktorovat response
 *
 * Class ImagesZone
 * @package Gallery\AdminModule\Components
 */
class ImagesZone extends BaseControl
{
	/** @var string */
	protected $uploadPath = '';

	/** @var Album */
	protected $album;

	/** @var Tags */
	protected $tagsService;

	/** @var IImageFinderFactory */
	protected $imageFinderFactory;

	/** @var callable[] */
	public $onBeforeUpload = [];

	/** @var callable[] */
	public $onEmpty = [];

	/** @var callable[] */
	public $onBeforeUpdate = [];

	/** @var callable[] */
	public $onAfterUpdate = [];

	/** @var callable[] */
	public $onPrepareDefaultImage = [];

	/** @var Images */
	protected $imagesService;

	/** @var array */
	public $extendedResponse = [];

	/** @var array */
	public $extendedResponseOnEmpty = [];

	/** @var bool */
	public $showImageFinder = true;

	/** @var Lang[] */
	public $contentLanguages = [];

	/** @var EntityManagerDecorator */
	public $em;

	protected array $customFieldsLatte = [
		'before' => [],
		'after'  => [],
	];

	/** @var array */
	public $data;

	public function __construct(Images $images, Tags $tags, IImageFinderFactory $imageFinderFactory, EntityManagerDecorator $em)
	{
		$this->imagesService      = $images;
		$this->tagsService        = $tags;
		$this->imageFinderFactory = $imageFinderFactory;
		$this->em                 = $em;
	}

	public function render()
	{
		$this->template->showImageFinder   = $this->showImageFinder;
		$this->template->album             = $this->album;
		$this->template->imagesJson        = $this->prepareDefaultImages();
		$this->template->lang              = $this->translator->getLocale();
		$this->template->contentLanguages  = $this->contentLanguages;
		$this->template->customFieldsLatte = $this->customFieldsLatte;
		$this->template->render(__DIR__ . '/ImagesZone.latte');
	}

	protected function attached($presenter): void
	{
		parent::attached($presenter);
		$this->contentLanguages = $this->langsService->getLangs(false);
	}

	/*******************************************************************************************************************
	 * =================  Handle
	 */

	public function handleGetTags()
	{
		$tags = [];
		foreach ($this->tagsService->getAvailableTags() as $tag)
			$tags[$tag->getId()] = $tag->title;

		$this->getPresenter()->payload->data = $tags;
		$this->getPresenter()->sendPayload();
	}

	public function handleTagsAutocomplete($q)
	{
		$tags = [];
		foreach ($this->tagsService->getAvailableTags() as $tag)
			$tags[] = ['key' => $tag->getId(), 'value' => $tag->title];

		$this->getPresenter()->sendJson($tags);
	}

	public function handleTagCreate($value)
	{
		$this->getPresenter()->sendJson($this->tagsService->createTag($value));
	}

	public function handleRemoveTag($value)
	{
		$responseData = ['status' => 'error', 'errorMessage' => ''];
		$tag          = $this->tagsService->findByTitle($value);

		if ($tag) {
			$this->tagsService->remove($tag->getId());
			$responseData['status'] = 'ok';
		}

		$this->getPresenter()->payload->data = $responseData;
		$this->getPresenter()->sendPayload();
	}

	public function handleOnEmpty()
	{
		$responseData        = ['status' => 'error', 'errorMessage' => ''];
		$this->data['title'] = $this->getPresenter()->getHttpRequest()->getPost('title');
		$this->onEmpty($this);

		if (!$this->album)
			$responseData['errorMessage'] = $this->t('gallery.image.albumMissing');
		else {
			$responseData['status']    = 'ok';
			$responseData['albumId']   = $this->album->getId();
			$responseData['albumPath'] = $this->album->generatePath();
		}

		$this->getPresenter()->payload->data = $responseData + $this->extendedResponseOnEmpty;
		$this->getPresenter()->sendPayload();
	}

	public function handleUpload()
	{
		$responseData = ['status' => 'error', 'errorMessage' => ''];
		$request      = $this->getPresenter()->getHttpRequest();
		$error        = false;

		$this->onBeforeUpload($this);

		try {
			$this->setAlbum($request->getPost('albumId'));
		} catch (\Exception $e) {
			$error                        = true;
			$responseData['errorMessage'] = $this->t('gallery.image.albumMissing');
		}

		if ($this->getUploadPath() == '') {
			$error                        = true;
			$responseData['errorMessage'] = $this->t('gallery.image.uploadPathMissing');
		}

		if (!$error) {
			foreach ($request->getFiles() as $file) {
				/** @var FileUpload $file */
				if ($file->isOk()) {
					$this->em->beginTransaction();
					try {
						$filename = $file->getSanitizedName();
						$image    = new Image($this->album, $filename);

						$this->em->persist($image);
						$this->em->flush();

						$filename        = $image->getId() . '_' . $filename;
						$image->filename = $filename;
						$this->em->persist($image);
						$this->em->flush();

						$file->move($this->getUploadDir() . '/' . $filename);

						$nImage = NImage::fromFile($image->getFile());
						if ($nImage->getWidth() > 1600 || $nImage->getHeight() > 1200) {
							$nImage->resize(1600, 1200, NImage::FIT);
							$nImage->save($image->getFile(), 100);
						}

						$uploadStatus = 'ok';
						$data         = [
							'albumPath'   => $this->album->generatePath(),
							'id'          => $image->getId(),
							'filename'    => $image->getFilename(),
							'dir'         => $this->album->generatePath(),
							'size'        => $image->getSize(),
							'link'        => $image->link,
							'source'      => [],
							'title'       => [],
							'description' => [],
							'tags'        => [],
							'isPublished' => $image->isPublished,
							'isCover'     => $image->isCover,
							'thumbCoords' => $image->getThumbCoords(),
						];

						foreach ($this->contentLanguages as $l) {
							$data['source'][$l->getTag()]      = $image->getText($l->getTag())->source;
							$data['title'][$l->getTag()]       = $image->getText($l->getTag())->title;
							$data['description'][$l->getTag()] = $image->getText($l->getTag())->description;
						}

						$this->em->commit();
					} catch (\Exception $e) {
						$this->em->rollback();
						@unlink($file->getTemporaryFile());

						if (isset($filename))
							@unlink($this->getUploadDir() . '/' . $filename);

						$uploadStatus = 'error';
						$data         = [];
					}
				} else {
					$uploadStatus = 'error';
					$data         = [];
				}

				$responseData['files'][$file->getName()] = [
					'status' => $uploadStatus,
					'data'   => $data,
				];

				$responseData['file'] = [
					'status' => $uploadStatus,
					'data'   => $data,
				];
			}

			if ($this->album) {
				$responseData['albumId'] = $this->album->getId();
			}

			$responseData['status'] = 'ok';
		}

		$this->getPresenter()->payload->data = $responseData + $this->extendedResponse;
		$this->getPresenter()->sendPayload();
	}

	public function handleUpdateImage($id)
	{
		$responseData = ['status' => 'error', 'errorMessage' => ''];
		$request      = $this->getPresenter()->getHttpRequest();
		/** @var Image $image */
		$image = $this->em->getRepository(Image::class)->find($id);
		$error = false;

		if (!$image) {
			$error = true;
		}

		if (!$error) {
			$data = $request->getPost('data');

			if (!$data) {
				$data[] = [
					'key'   => $request->getPost('key', null),
					'value' => $request->getPost('value', null),
					'lang'  => $request->getPost('lang', null),
				];
			}

			$this->em->beginTransaction();
			try {
				foreach ($data as $v) {
					$this->onBeforeUpdate($id, $v);
					$key   = $v['key'];
					$value = $v['value'];
					$lang  = $v['lang'];

					if ($key && $value !== null && property_exists(Image::class, $key)) {
						if ($key == 'isCover' && $value == 1)
							$this->imagesService->setAsCover($image->getId());
						elseif ($key == 'tags') {
							$tags = $value == '' ? [] : array_map(function($id) {
								return $this->tagsService->getReference($id);
							}, explode(',', $value));
							$image->setTags($tags);
						} elseif ($lang) {
							$txt = $image->getText($lang);
							if (!$txt) {
								$txt = new ImageText($image, $lang);
								$image->addText($txt);
							}
							$txt->$key = $value;
							$this->em->persist($txt);
						} else
							$image->$key = $value;

						$this->em->persist($image);
						$this->onAfterUpdate($image);

						$responseData[$key]['status'] = 'ok';
					}
				}
				$this->em->flush();
				$this->em->commit();
				$responseData['status'] = 'ok';
			} catch (\Exception $e) {
				$responseData['status'] = 'error';
				$this->em->rollback();
			}
		}

		$this->getPresenter()->payload->data = $responseData + $this->extendedResponse;
		$this->getPresenter()->sendPayload();
	}

	public function handleChangePosition($id)
	{
		$responseData = ['status' => 'error', 'errorMessage' => ''];
		$request      = $this->getPresenter()->getHttpRequest();
		$error        = false;
		/** @var Image $image */
		$image = $this->em->getRepository(Image::class)->find($id);

		if (!$image) {
			$error = true;
		}

		if (!$error && ($position = $request->getPost('position', null)) != null) {
			$image->setPosition((int) $position);
			$this->em->persist($image);
			$this->em->flush();

			$responseData['status'] = 'ok';
		}


		$this->getPresenter()->payload->data = $responseData + $this->extendedResponse;
		$this->getPresenter()->sendPayload();
	}

	public function handleRemoveImage()
	{
		$responseData = ['status' => 'error', 'errorMessage' => ''];
		$request      = $this->getPresenter()->getHttpRequest();
		$id           = $request->getPost('id', null);

		if ($this->imagesService->removeImage($id)) {
			$responseData['status'] = 'ok';
		}

		$this->getPresenter()->payload->data = $responseData + $this->extendedResponse;
		$this->getPresenter()->sendPayload();
	}

	public function handleUseFoundImage()
	{
		/** @var Image $image */
		/** @var Image $img */
		$presenter = $this->getPresenter();
		try {
			$imgId   = $presenter->request->getPost('imgId');
			$albumId = $presenter->request->getPost('albumId');

			if (!$imgId || !$albumId)
				throw new \InvalidArgumentException();

			$img = $this->imagesService->get($imgId);

			if ($img) {
				$image = clone $img;
				$image->setAlbum($this->em->getReference(Album::class, $albumId));
				$image->setCloneOf($img);
				$image->clear();
				$img->setLastUse();

				$this->em->persist($img);
				$this->em->persist($image);

				foreach ($img->getTexts()->toArray() as $t) {
					/** @var ImageText $t */
					$it              = new ImageText($image, $t->getLang());
					$it->title       = $t->title;
					$it->description = $t->description;
					$it->source      = $t->source;

					$this->em->persist($it);
				}

				$this->em->flush();

				$data = [
					'id'          => $image->getId(),
					'filename'    => $image->getFilename(),
					'dir'         => $image->path,
					'size'        => $image->getSize(),
					'source'      => [],
					'title'       => [],
					'description' => [],
					'link'        => $image->link,
					'tags'        => [],
					'isPublished' => $image->isPublished,
					'isCover'     => $image->isCover,
					'thumbCoords' => $image->getThumbCoords(),
				];

				foreach ($this->contentLanguages as $l) {
					$data['source'][$l->getTag()]      = $image->getText($l->getTag())->source;
					$data['title'][$l->getTag()]       = $image->getText($l->getTag())->title;
					$data['description'][$l->getTag()] = $image->getText($l->getTag())->description;
				}

				$presenter->payload->image = $data;

				$presenter->flashMessageSuccess('gallery.imageZone.imageAdded');
			}
		} catch (\Exception $e) {
			$presenter->flashMessageDanger('gallery.imageZone.imageAddFailed');
			throw $e;
		}

		$presenter->redrawControl('flashes');
	}

	/*******************************************************************************************************************
	 * =================  Get/Set
	 */

	private function prepareDefaultImages()
	{
		$data = [];

		if ($this->album) {
			foreach ($this->album->getImages() as $image) {
				/** @var Image $image */
				$arr = [
					'id'          => $image->getId(),
					'filename'    => $image->getFilename(),
					'dir'         => $image->path,
					'size'        => $image->getSize(),
					'source'      => [],
					'title'       => [],
					'description' => [],
					'link'        => $image->link,
					'tags'        => array_map(function($arr) {
						return ['key' => $arr->getId(), 'value' => $arr->title];
					}, $image->getTags()->toArray()),
					'isPublished' => $image->isPublished,
					'isCover'     => $image->isCover,
					'thumbCoords' => $image->getThumbCoords(),
				];

				foreach ($this->contentLanguages as $l) {
					$arr['source'][$l->getTag()]      = $image->getText($l->getTag())->source;
					$arr['title'][$l->getTag()]       = $image->getText($l->getTag())->title;
					$arr['description'][$l->getTag()] = $image->getText($l->getTag())->description;
				}

				foreach ($this->onPrepareDefaultImage as $c)
					$c($image, $arr);

				$data[] = $arr;
			}
		}

		return Json::encode($data);
	}

	public function setUploadPath($path)
	{
		if ($path != '')
			$this->uploadPath = $path;

		return $this;
	}

	public function getUploadPath() { return $this->uploadPath; }

	public function getUploadDir() { return WWW_DIR . '/' . $this->getUploadPath(); }

	public function setAlbum($id)
	{
		$this->album = $this->em->getRepository(Album::class)->createQueryBuilder('a')->where('a.id = :id')->setParameter('id', $id)
			->leftJoin('a.images', 'imgs')->leftJoin('imgs.tags', 'tags')
			->addSelect('imgs, tags')->getQuery()->getOneOrNullResult();

		if (!$this->album)
			$this->getPresenter()->error(null, IResponse::S404_NOT_FOUND);
		else {
			$this->setUploadPath($this->album->generatePath());
			if ($this->showImageFinder)
				$this['imageFinder']->setActiveAlbum($this->album);
		}
	}

	public function getEm()
	{
		return $this->em;
	}

	public function addCustomFieldLatteBefore(string $file): self
	{
		$this->customFieldsLatte['before'][] = $file;

		return $this;
	}

	public function addCustomFieldLatteAfter(string $file): self
	{
		$this->customFieldsLatte['after'][] = $file;

		return $this;
	}

	/*******************************************************************************************************************
	 * =================  Components
	 */

	protected function createComponentImageFinder()
	{
		$control = $this->imageFinderFactory->create();

		return $control;
	}

	public function handleShowGallery($albumId = null)
	{
		if ($albumId)
			$this->setAlbum($albumId);
		$this->getPresenter()->getTemplate()->modal            = 'test';
		$this->getPresenter()->getTemplate()->modalDialogClass = 'modal-full';
		ob_start();
		$this->render();
		$body                                           = ob_get_clean();
		$this->getPresenter()->getTemplate()->modalBody = Html::el()->addHtml($body);
		$this->getPresenter()->redrawControl('modal');
	}
}
