<?php declare(strict_types = 1);

namespace Gallery\AdminModule\Model;

use Core\Model\Helpers\BaseEntityService;
use Core\Model\Images\ImagePipe;
use Gallery\Model\Entities\Image;

/**
 * Class Images
 * @package Blog\Model
 *
 * @method Image|null|object = getReference($id)
 * @method Image[]|null getAll()
 * @method Image|null get($id)
 */
class Images extends BaseEntityService
{
	protected $entityClass = Image::class;

	/** @var ImagePipe */
	protected $imagePipe;

	public function __construct(ImagePipe $imagePipe)
	{
		$this->imagePipe = $imagePipe;
	}

	/**
	 * @return Image[]
	 */
	public function getImages()
	{
		$qb = $this->getEr()->createQueryBuilder('i')
			->addSelect('it')
			->leftJoin('i.texts', 'it')
			->orderBy('i.position', 'DESC');

		return $qb->getQuery()->getResult();
	}

	/**
	 * @param $id
	 *
	 * @return Image|null
	 */
	public function getImage($id)
	{
		return $this->getEr()->createQueryBuilder('i')
			->addSelect('it')
			->leftJoin('i.texts', 'it')
			->where('i.id = :id')->setParameter('id', $id)
			->getQuery()->getOneOrNullResult();
	}

	public function getCoverByAlbum($albumId)
	{
		return $this->getEr()->createQueryBuilder('i')->andWhere('i.isCover = 1 OR (i.isPublished = 1 AND i.isCover = 0)')
			->andWhere('i.album = :album')->setParameter('album', $albumId)->orderBy('i.position', 'ASC')
			->getQuery()->setMaxResults(1)->getOneOrNullResult();
	}

	/**
	 * @param $id
	 *
	 * @return bool
	 * @throws \Exception
	 */
	public function setAsCover($id)
	{
		if ($image = $this->getEr()->find($id)) {
			foreach ($this->getEr()->findBy(['isCover' => 1, 'album' => $image->getAlbum()->getId()]) as $img) {
				$img->isCover = 0;
				$this->em->persist($img);
			}

			$image->isCover = 1;
			$this->em->persist($image);

			$this->em->flush();

			return true;
		}

		return false;
	}

	/**
	 * @param array|int $id
	 *
	 * @return array|bool
	 * @throws \Exception
	 */
	public function remove($id)
	{
		if ($image = $this->getEr()->find($id)) {
			/** @var Image $image */
			$same = $this->getEr()->createQueryBuilder('i')
				->where('i.filename = :filename')
				->andWhere('i.id != :id')
				->setParameters([
					'filename' => $image->getFilename(),
					'id'       => $id,
				])->getQuery()->getArrayResult();
			if (empty($same) && file_exists($image->getFile()))
				unlink($image->getFile());
			$this->imagePipe->removeThumbs($image->getFilePath());
			$this->em->remove($image);
			$this->em->flush();

			return true;
		}

		return false;
	}

	public function search($q)
	{
		$values = array_map(function($v) {
			return trim($v);
		}, explode(',', $q));

		$qr = $this->getEr()->createQueryBuilder('i');

		foreach ($values as $k => $v) {
			$qr->orWhere('i.description LIKE :k' . $k)->orWhere('i.source LIKE :k' . $k)->orWhere('i.filename LIKE :k' . $k)
				->orWhere('i.title LIKE :k' . $k)->orWhere('i.path LIKE :k' . $k)
				->leftJoin('i.texts', 'it')->addSelect('it')
				->leftJoin('i.tags', 'tags')->orWhere('tags.title LIKE :k' . $k)
				->leftJoin('i.album', 'a')->addSelect('a')
				->groupBy('i.id')
				->setParameter('k' . $k, "%$v%");
		}

		$images = [];
		foreach ($qr->getQuery()->getResult() as $image) {
			/** @var Image $image */
			$key = $image->getCloneOf() !== null ? $image->getCloneOf()->getId() : $image->getId();

			if (isset($images[$key]) && $image->getLastUse() && $images[$key]->getLastUse() && $image->getLastUse()->getTimestamp() < $images[$key]->getLastUse()->getTimestamp()) {
				continue;
			}

			$images[$key] = $image;
		}

		usort($images, function($a, $b) {
			$aLast = $a->getLastUse() ? $a->getLastUse()->getTimestamp() : null;
			$bLast = $b->getLastUse() ? $b->getLastUse()->getTimestamp() : null;

			return $aLast <=> $bLast;
		});

		$images = array_reverse($images);

		return $images;
	}

	/**
	 * @param $id
	 *
	 * @return Image[]|null
	 */
	public function getByAlbum($id)
	{
		$qb = $this->getEr()->createQueryBuilder('i')
			->addSelect('it')
			->leftJoin('i.texts', 'i')
			->where('i.album = :album')->setParameter('album', $id)
			->orderBy('i.position');

		return $qb->getQuery()->getResult();
	}

	/**
	 * @param $id
	 *
	 * @return bool
	 * @throws \Exception
	 */
	public function removeImage($id)
	{
		return $this->remove($id);
	}
}
