<?php declare(strict_types = 1);

namespace Gallery\FrontModule\Model;

use Core\Model\Helpers\BaseFrontEntityService;
use Core\Model\Helpers\DaoHelper;
use Core\Model\Lang\DefaultLang;
use Doctrine\ORM\Query;
use Gallery\Model\Entities\Album;
use Nette\Caching\Cache;
use Gallery\FrontModule\Model\Dao;

/**
 * Class Albums
 * @package Gallery\FrontModule\Model
 */
class Albums extends BaseFrontEntityService
{
	/** @var DefaultLang @inject */
	public $defaultLang;

	/** @var Images @inject */
	public $imagesService;

	const CACHE_NAMESPACE = 'albums';

	protected $entityClass = Album::class;

	/** @var array */
	protected $cacheDep = [
		Cache::TAGS    => ['albums'],
		Cache::EXPIRE  => '1 week',
		Cache::SLIDING => true,
	];

	public function getCache()
	{
		if ($this->cache === null)
			$this->cache = new Cache($this->cacheStorage, self::CACHE_NAMESPACE);

		return $this->cache;
	}

	/**
	 * TODO cache
	 *
	 * @param int $id
	 *
	 * @return Dao\Album[]|null
	 * @throws \Doctrine\ORM\NonUniqueResultException
	 */
	public function get($id)
	{
		$locale = $this->defaultLang->locale;
		$key    = 'album_' . $locale . '_' . $id;

		return $this->getCache()->load($key, function(&$dep) use ($id, $locale) {
			$dep                = $this->cacheDep;
			$dep[Cache::TAGS][] = 'album/' . $id;

			$album = $this->getEr()->createQueryBuilder('a', 'a.id')
				->addSelect('at')
				->andWhere('a.id = :id')->setParameter('id', $id)
				->leftJoin('a.texts', 'at', 'WITH', 'at.lang = :lang')
				->setParameter('lang', $locale)
				->andWhere('a.isPublished = 1')
				->groupBy('a.id')
				->getQuery()->getOneOrNullResult(Query::HYDRATE_ARRAY);

			return $album ? $this->fillDao($album) : null;
		});
	}

	/**
	 * @param array $a
	 *
	 * @return Dao\Album
	 */
	protected function fillDao($a): Dao\Album
	{
		$locale = $this->defaultLang->locale;

		$album              = new Dao\Album();
		$album->id          = $a['id'];
		$album->title       = $a['texts'][$locale]['title'];
		$album->description = $a['texts'][$locale]['description'];

		$imgs  = $this->imagesService->getImagesByAlbum($a['id']);
		$cover = null;

		foreach ($imgs as $k => $i)
			if ($i->isCover) {
				$cover = $i;
				break;
			}

		if (!$cover && $imgs) {
			$cover          = reset($imgs);
			$cover->isCover = 1;
		}

		if ($cover)
			$album->setCover($cover);
		$album->setImages($imgs);

		return $album;
	}
}
