<?php declare(strict_types = 1);

namespace Gallery\FrontModule\Model;

use Core\Model\Helpers\BaseFrontEntityService;
use Gallery\Model\Entities\Image;
use Gallery\FrontModule\Model\Dao;

/**
 * Class Images
 * @package Gallery\FrontModule\Model
 */
class Images extends BaseFrontEntityService
{
	const CACHE_NAMESPACE = 'images';

	protected $entityClass = Image::class;

	/** @var Dao\Image[][] */
	protected $cImages = [];

	public function getCoverImage(int $albumId): ?Dao\Image
	{
		if (!$albumId)
			return null;

		$imgs = $this->getImagesByAlbum($albumId);
		foreach ($imgs as $img)
			if ($img->isCover)
				return $img;

		return $imgs[0] ?? null;
	}

	/**
	 * @param int $albumId
	 *
	 * @return Dao\Image[]
	 */
	public function getImagesByAlbum(int $albumId): array
	{
		if (!isset($this->cImages[$albumId])) {
			$images = $this->getEr()->createQueryBuilder('i', 'i.id')
				->addSelect('it')
				->where('i.isPublished = 1')
				->andWhere('i.album = :album')->setParameter('album', $albumId)
				->leftJoin('i.texts', 'it', 'WITH', 'it.lang = :lang')
				->setParameter('lang', $this->translator->getLocale())
				->orderBy('i.position')
				->groupBy('i.id')
				->getQuery()->getArrayResult();

			$this->cImages[$albumId] = $this->fillDao($images);
		};

		return $this->cImages[$albumId];
	}

	/**
	 * @param array $images
	 *
	 * @return Dao\Image[]
	 */
	protected function fillDao($images): array
	{
		$imgs   = [];
		$locale = $this->translator->getLocale();

		foreach ($images as $img) {
			$i              = new Dao\Image();
			$i->id          = $img['id'];
			$i->title       = $img['texts'][$locale]['title'];
			$i->description = $img['texts'][$locale]['description'];
			$i->source      = $img['texts'][$locale]['source'];
			$i->link        = $img['texts'][$locale]['link'];
			$i->position    = $img['position'];
			$i->isCover     = $img['isCover'];
			$i->thumbCoords = $img['thumbCoords'];
			$i->setFile($img['path'] . DIRECTORY_SEPARATOR . $img['filename']);

			$imgs[$img['id']] = $i;
		}

		return $imgs;
	}
}
