<?php declare(strict_types = 1);

namespace Gallery\AdminModule\Components\Album;

use Core\AdminModule\Components\ContentLangSwitcher\IContentLangSwitcherFactory;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Gallery\AdminModule\Components\Image\IImagesZoneFactory;
use Gallery\AdminModule\Components\Image\ImagesZone;
use Gallery\AdminModule\Model\Albums;
use Gallery\Model\Entities\Album;
use Gallery\Model\Entities\AlbumText;
use Nette\Http\IResponse;
use Nette\Utils\ArrayHash;
use Nette\Utils\DateTime;

class AlbumForm extends BaseControl
{
	/** @var IImagesZoneFactory */
	protected $imagesZoneFactory;

	/** @var Album */
	protected $album;

	/** @var Albums */
	protected $albumsService;

	/** @var IContentLangSwitcherFactory */
	protected $contentLangSwitcherFactory;

	private $albumBasePath = UPLOADS_PATH . '/gallery';

	public function __construct(IImagesZoneFactory $imagesZoneFactory, Albums $albums, IContentLangSwitcherFactory $contentLangSwitcherFactory)
	{
		$this->imagesZoneFactory          = $imagesZoneFactory;
		$this->albumsService              = $albums;
		$this->contentLangSwitcherFactory = $contentLangSwitcherFactory;
	}

	public function render()
	{
		$this->template->render(__DIR__ . '/AlbumForm.latte');
	}

	protected function createComponentForm()
	{
		$form = $this->createForm();

		$form->addText('title', 'gallery.albumForm.title')->setMaxLength(255)->setIsMultilanguage();
		$form->addTextArea('description', 'gallery.albumForm.description')->setIsMultilanguage();
		$form->addBool('isPublished', 'gallery.albumForm.isPublished')->setDefaultValue(1);

		$form->addHidden('preparedAlbumId');
		$form->addSaveCancelControl();

		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values)
	{
		try {
			$presenter = $this->getPresenter();
			$user      = $presenter->getUser();

			if ($this->album)
				$album = $this->album;
			else if ($values->preparedAlbumId)
				$album = $this->albumsService->getAlbum($values->preparedAlbumId);
			else
				$album = new Album($this->albumBasePath);

			$langs      = [];
			$langInputs = ['title', 'description'];
			foreach ($langInputs as $name)
				foreach ($values->$name as $l => $v)
					if ($v != '')
						$langs[$l] = $l;

			foreach ($langs as $l) {
				$texts = $album->getText($l);

				if (!$texts)
					$texts = new AlbumText($album, $l);

				foreach ($langInputs as $name)
					$texts->$name = $values->$name[$l];
				$this->em->persist($texts);
			}

			$album->isPublished = $values->isPublished;

			$this->em->persist($album);
			$this->em->flush();
			$form->addCustomData('albumId', $album->getId());
		} catch (\Exception $e) {
			$form->addError($e->getMessage());

			return false;
		}
	}

	/*******************************************************************************************************************
	 * =================  Get/Set
	 */

	public function setAlbum($id)
	{
		$this->album = $this->albumsService->getAlbum($id);

		if (!$this->album)
			$this->getPresenter()->error(null, IResponse::S404_NOT_FOUND);

		$defaults = [];
		foreach ($this->langsService->getLangs(false) as $lang) {
			foreach (['title', 'description'] as $name) {
				$defaults[$name][$lang->getTag()] = $this->album->getText($lang->getTag())->$name;
			}
		}

		$this['form']->setDefaults($defaults);
	}

	/*******************************************************************************************************************
	 * ============================== Components
	 */

	protected function createComponentImagesZone()
	{
		$control = $this->imagesZoneFactory->create();

		if ($this->album) {
			$control->setAlbum($this->album->getId());
		}

		$control->onEmpty[] = function($control) {
			/** @var ImagesZone $control */

			if (!$this->album) {
				$album        = new Album($this->albumBasePath);
				$album->title = $this->t('gallery.album.withoutName') . ' - ' . (new DateTime())->format('j. n. Y');
				$this->em->persist($album);
				$this->em->flush();

				$control->setAlbum($album->getId());

				$control->extendedResponseOnEmpty['albumId'] = $album->getId();
			}
		};

		return $control;
	}

	protected function createComponentContentLangSwitcher()
	{
		$control = $this->contentLangSwitcherFactory->create();

		$control->templateFile = 'Inline';

		return $control;
	}
}
