<?php declare(strict_types = 1);

namespace Gallery\AdminModule\Model;

use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Query\Parameter;
use Core\Model\Helpers\BaseEntityService;
use Core\Model\Images\ImagePipe;
use Doctrine\ORM\NonUniqueResultException;
use Exception;
use Gallery\Model\Entities\Image;

/**
 * @method Image|null getReference($id)
 * @method Image[] getAll()
 * @method Image|null get($id)
 */
class Images extends BaseEntityService
{
	protected $entityClass = Image::class;

	public function __construct(protected ImagePipe $imagePipe)
	{
	}

	/**
	 * @return Image[]
	 */
	public function getImages(): array
	{
		$qb = $this->getEr()->createQueryBuilder('i')
			->addSelect('it')
			->leftJoin('i.texts', 'it')
			->orderBy('i.position', 'DESC');

		return $qb->getQuery()->getResult();
	}

	/**
	 * @throws NonUniqueResultException
	 */
	public function getImage(int|string $id): ?Image
	{
		return $this->getEr()->createQueryBuilder('i')
			->addSelect('it')
			->leftJoin('i.texts', 'it')
			->where('i.id = :id')->setParameter('id', $id)
			->getQuery()->getOneOrNullResult();
	}

	/**
	 * @throws NonUniqueResultException
	 */
	public function getCoverByAlbum(int|string $albumId): ?Image
	{
		return $this->getEr()
			->createQueryBuilder('i')
			->andWhere('i.isCover = 1 OR (i.isPublished = 1 AND i.isCover = 0)')
			->andWhere('i.album = :album')
			->setParameter('album', $albumId)
			->orderBy('i.position', 'ASC')
			->getQuery()
			->setMaxResults(1)
			->getOneOrNullResult();
	}

	/**
	 * @throws Exception
	 */
	public function setAsCover(int|string $id): bool
	{
		if ($image = $this->getEr()->find($id)) {
			foreach ($this->getEr()->findBy(['isCover' => 1, 'album' => $image->getAlbum()->getId()]) as $img) {
				$img->isCover = 0;
				$this->em->persist($img);
			}

			$image->isCover = 1;
			$this->em->persist($image);

			$this->em->flush();

			return true;
		}

		return false;
	}

	/**
	 * @param array|int $id
	 *
	 * @throws Exception
	 */
	public function remove($id): bool
	{
		if ($image = $this->getEr()->find($id)) {
			/** @var Image $image */
			$same = $this->getEr()->createQueryBuilder('i')
				->where('i.filename = :filename')
				->andWhere('i.id != :id')
				->setParameters(new ArrayCollection([new Parameter('filename', $image->getFilename()), new Parameter('id', $id)]))->getQuery()->getArrayResult();

			if (empty($same) && file_exists($image->getFile())) {
				unlink($image->getFile());
			}
			$this->imagePipe->removeThumbs($image->getFilePath());
			$this->em->remove($image);
			$this->em->flush();

			return true;
		}

		return false;
	}

	/**
	 * @return Image[]
	 */
	public function search(string $q): array
	{
		$values = array_map('trim', explode(',', $q));

		$qr = $this->getEr()->createQueryBuilder('i');

		foreach ($values as $k => $v) {
			$qr->orWhere('i.description LIKE :k' . $k)
				->orWhere('i.source LIKE :k' . $k)
				->orWhere('i.filename LIKE :k' . $k)
				->orWhere('i.title LIKE :k' . $k)
				->orWhere('i.path LIKE :k' . $k)
				->leftJoin('i.texts', 'it')
				->addSelect('it')
				->leftJoin('i.tags', 'tags')
				->orWhere('tags.title LIKE :k' . $k)
				->leftJoin('i.album', 'a')
				->addSelect('a')
				->groupBy('i.id')
				->setParameter('k' . $k, "%$v%");
		}

		$images = [];
		foreach ($qr->getQuery()->getResult() as $image) {
			/** @var Image $image */
			$key = $image->getCloneOf() !== null ? $image->getCloneOf()->getId() : $image->getId();

			if (isset($images[$key]) && $image->getLastUse() && $images[$key]->getLastUse() && $image->getLastUse()
					->getTimestamp() < $images[$key]->getLastUse()->getTimestamp()) {
				continue;
			}

			$images[$key] = $image;
		}

		usort($images, static function(Image $a, Image $b) {
			$aLast = $a->getLastUse() ? $a->getLastUse()->getTimestamp() : null;
			$bLast = $b->getLastUse() ? $b->getLastUse()->getTimestamp() : null;

			return $aLast <=> $bLast;
		});

		return array_reverse($images);
	}

	/**
	 * @return Image[]
	 */
	public function getByAlbum(int|string $id): array
	{
		$qb = $this->getEr()->createQueryBuilder('i')
			->addSelect('it')
			->leftJoin('i.texts', 'i')
			->where('i.album = :album')->setParameter('album', $id)
			->orderBy('i.position');

		return $qb->getQuery()->getResult();
	}

	/**
	 * @throws Exception
	 */
	public function removeImage(int|string $id): bool
	{
		return $this->remove($id);
	}
}
