<?php declare(strict_types = 1);

namespace Gallery\FrontModule\Model;

use Core\Model\Application\AppState;
use Core\Model\Dao\Site;
use Core\Model\Helpers\BaseFrontEntityService;
use Core\Model\Helpers\Strings;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Query\Parameter;
use Gallery\FrontModule\Model\Dao\Image;
use Gallery\Model\Entities\Album;
use Gallery\Model\GalleryConfig;
use Nette\Caching\Cache;

class Albums extends BaseFrontEntityService
{
	public const CACHE_NAMESPACE = 'albums';

	public ?array $placeholderImageData = null;

	protected       $entityClass = Album::class;
	protected array $cAlbums     = [];
	protected array $cacheDep    = [
		Cache::Tags   => ['albums'],
		Cache::Expire => '1 week',
	];

	public function __construct(
		protected Images $imagesService,
	)
	{
	}

	public function getCache(): Cache
	{
		if ($this->cache === null) {
			$this->cache = new Cache($this->cacheStorage, self::CACHE_NAMESPACE);
		}

		return $this->cache;
	}

	/**
	 * @return Dao\Album[]
	 */
	public function getAlbums(array $ids, ?string $usePlaceholder = null): array
	{
		$whereIds = [];
		$result   = [];
		$locale   = $this->translator->getLocale();

		$keys = [];
		foreach ($ids as $aId) {
			if (isset($this->cAlbums[$aId])) {
				$result[$aId] = $this->cAlbums[$aId];
			} else {
				$keys[] = 'album/' . $locale . '/' . $aId;
			}
		}

		if (!empty($keys)) {
			foreach ($this->getCache()->bulkLoad($keys) as $key => $album) {
				$tmp = explode('/', $key);
				$id  = end($tmp);

				if ($album) {
					$result[$id]        = $album;
					$this->cAlbums[$id] = $album;
				} else {
					$whereIds[]  = $id;
					$result[$id] = null;
				}
			}
		}

		$loadImages = [];
		if (!empty($whereIds)) {
			foreach ($this->getEr()->createQueryBuilder('a', 'a.id')
				         ->addSelect('at')
				         ->andWhere('a.id IN (:ids)')
				         ->leftJoin('a.texts', 'at', 'WITH', 'at.lang = :lang')
				         ->andWhere('a.isPublished = 1')
				         ->groupBy('a.id')
				         ->setParameters(new ArrayCollection([new Parameter('ids', $whereIds), new Parameter('lang', $locale)]))
				         ->getQuery()->getArrayResult() as $row) {
				$dao                     = $this->fillDao($row);
				$result[$dao->id]        = $dao;
				$this->cAlbums[$dao->id] = $dao;
				$loadImages[$dao->id]    = $result[$dao->id];
			}
		}

		if (!empty($loadImages)) {
			$this->setImages($loadImages, $usePlaceholder);

			foreach ($loadImages as $album) {
				$cacheDep                = $this->cacheDep;
				$cacheDep[Cache::Tags][] = 'album/' . $album->id;
				$this->getCache()->save('album/' . $locale . '/' . $album->id, $album, $cacheDep);
			}
		}

		if ($usePlaceholder) {
			foreach ($result as $id => $album) {
				$album->placeholderCallback = function() use ($usePlaceholder) {
					return $this->getPlaceholderImageData($usePlaceholder);
				};
			}
		}

		return $result;
	}

	/**
	 * @param int|string $id
	 *
	 * @return Dao\Album|null
	 */
	public function get($id): ?Dao\Album
	{
		$id = (int) $id;

		return $this->getAlbums([$id])[$id] ?? null;
	}

	protected function fillDao(array $a): Dao\Album
	{
		$locale = $this->translator->getLocale();

		$album              = new Dao\Album;
		$album->id          = $a['id'];
		$album->title       = $a['texts'][$locale]['title'];
		$album->description = $a['texts'][$locale]['description'];

		return $album;
	}

	/**
	 * @param Dao\Album[] $albums
	 */
	protected function setImages(array &$albums, ?string $usePlaceholder = null): void
	{
		$albumIds = [];
		foreach ($albums as $album) {
			$albumIds[] = $album->id;
		}

		$allImages = $this->imagesService->getImagesByAlbum($albumIds);

		foreach ($albums as &$album) {
			$cover = null;
			$imgs  = $allImages[$album->id] ?? [];

			if ($usePlaceholder && empty($imgs)) {
				$tmp     = new Image();
				$tmp->id = 0;

				$imgs[] = $tmp;
			}

			foreach ($imgs as $k => $i) {
				$i->album = $album;

				if ($i->isCover) {
					$cover = $i;
					unset($imgs[$k]);
				}
			}

			if (!$cover && $imgs) {
				/** @var Dao\Image $cover */
				$cover          = reset($imgs);
				$cover->isCover = 1;
			}

			if ($cover) {
				$album->setCover($cover);
				$album->defaultCover = $cover;
			}

			$album->setImages(array_merge([$cover], $imgs));
		}
	}

	public function getPlaceholderImageData(string $key): array
	{
		if ($this->placeholderImageData === null) {
			/** @var Site $currentSite */
			$currentSite = AppState::getState('currentSite');
			$key         = str_replace('%siteIdent%', Strings::firstUpper($currentSite->getIdent()), $key);
			$data        = $this->settings->get($key);

			if ($data[$key . '_file'] ?? null) {
				$this->placeholderImageData = [
					'file'  => $data[$key . '_file'] ?? null,
					'alt'   => $data[$key . '_alt'] ?? null,
					'title' => $data[$key . '_title'] ?? null,
				];
			} else {
				$this->placeholderImageData = [
					'file'  => GalleryConfig::load('image.notFoundImage'),
					'alt'   => null,
					'title' => null,
				];
			}
		}

		return $this->placeholderImageData;
	}
}
