<?php declare(strict_types = 1);

namespace Gallery\Model;

use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Helpers\Strings;
use Core\Model\Images\ImageHelper;
use Doctrine\ORM\Query\Expr\Join;
use Gallery\Model\Entities\Album;
use Gallery\Model\Entities\Image;
use Nette\Utils\FileSystem;

class Cleaner
{
	public function __construct(
		protected Albums                 $albums,
		protected Images                 $images,
		protected EntityManagerDecorator $em,
	)
	{
	}

	public function runCheckImageDuplicity(): void
	{
		$list = $this->checkImageDuplicity();

		if ($list) {
			$this->removeImageDuplicityFromDb($list);
		}
	}

	public function autoResizeGalleries(): void
	{
		foreach ($this->images->getEr()->createQueryBuilder('i')
			         ->select('i.id, i.filename, i.path')->getQuery()->getScalarResult() as $row) {
			$path = WWW_DIR . $row['path'] . DS . $row['filename'];
			ImageHelper::autoResize($path);
		}
	}

	public function checkEmptyAlbums(): array
	{
		$arr = [];

		foreach ($this->em->getRepository(Album::class)->createQueryBuilder('a')
			         ->select('a.id')
			         ->leftJoin('a.images', 'i', Join::WITH, 'a.id = i.album')
			         ->where('i.id IS NULL')
			         ->getQuery()->getScalarResult() as $row) {
			$arr[$row['id']] = $row['id'];
		}

		file_put_contents(LOG_DIR . DS . '_gallery_emptyAlbums_ids.log', implode(",", array_keys($arr)));

		return $arr;
	}

	public function checkNotFoundImages(): array
	{
		$arr = [];

		foreach ($this->images->getEr()->createQueryBuilder('i')
			         ->select('i.id, i.filename, i.path')->getQuery()->getScalarResult() as $row) {
			$path = WWW_DIR . $row['path'] . DS . $row['filename'];
			if (!file_exists($path))
				$arr[$row['id']] = str_pad((string) $row['id'], 6, '0', STR_PAD_LEFT) . ' | ' . $path;
		}

		file_put_contents(LOG_DIR . DS . '_gallery_notFoundImagesFromGallery.log', implode("\r", $arr));
		file_put_contents(LOG_DIR . DS . '_gallery_notFoundImagesFromGallery_ids.log', implode(",", array_keys($arr)));

		return $arr;
	}

	public function checkFilesNotInImages(): array
	{
		$allImages = [];
		$arr       = [];

		foreach ($this->images->getEr()->createQueryBuilder('i')
			         ->select('i.id, i.filename, i.path')->getQuery()->getScalarResult() as $row) {
			$path = realpath(WWW_DIR . $row['path'] . DS . $row['filename']);
			if ($path) {
				$allImages[$path] = $row['id'];
			}
		}

		foreach (glob(UPLOADS_DIR . DS . 'products' . DS . '**' . DS . '*.*') as $file) {
			$realpath = realpath($file);

			if (!isset($allImages[$realpath])) {
				$arr[] = $realpath;
				continue;
			}

			if (filesize($realpath) === 0) {
				$arr[] = $realpath;
			}
		}

		file_put_contents(LOG_DIR . DS . '_gallery_filesNotInImages.log', implode("\r", $arr));

		return $arr;
	}

	public function removeImagesFromDB(array $ids): void
	{
		foreach ($ids as $id) {
			/** @var Image $image */
			$image = $this->images->getReference($id);
			$this->em->remove($image);
		}

		$this->em->flush();
	}

	public function removeAlbumsFromDb(array $ids): void
	{
		$this->albums->removeMultiple($ids);
	}

	public function removeImages(array $list): void
	{
		foreach ($list as $row) {
			FileSystem::delete($row);
		}
	}

	public function checkImageDuplicity(): array
	{
		$arr = [];

		foreach ($this->em->getConnection()->fetchAllAssociative("SELECT GROUP_CONCAT(id) as ids, album_id 
					FROM `gallery__image` 
					WHERE clone_of_id IS NULL
					GROUP BY album_id, CONCAT(path, filename) 
				    HAVING COUNT(id) > 1") as $row) {
			$arr[] = explode(',', $row['ids']);
		}

		file_put_contents(LOG_DIR . DS . '_gallery_checkImageDuplicity_ids.log', implode(",", array_keys($arr)));

		return $arr;
	}

	public function removeImageDuplicityFromDb(array $data): void
	{
		foreach ($data as $images) {
			array_shift($images);
			$this->em->getConnection()->executeQuery("DELETE FROM `gallery__image` WHERE id IN (" . implode(',', $images) . ")");
		}
	}

	public function checkImageNames(): array
	{
		$arr  = [];
		$conn = $this->em->getConnection();

		foreach ($conn->fetchAllAssociative("SELECT id, filename, path, clone_of_id 
					FROM `gallery__image`") as $row) {
			$newFileName = Strings::webalize($row['filename'], '_.', false);

			if ($newFileName !== $row['filename']) {
				$fullPath    = WWW_DIR . $row['path'] . DS . $row['filename'];
				$fullPathNew = WWW_DIR . $row['path'] . DS . $newFileName;

				if (file_exists($fullPath)) {
					FileSystem::rename($fullPath, $fullPathNew);

					if (file_exists($fullPathNew)) {
						$conn->update('gallery__image', [
							'filename' => $newFileName,
						], [
							'id' => $row['id'],
						]);
					}
				}

				$arr[] = $row['id'] . ' | ' . $row['filename'] . ' | ' . $newFileName;
			}
		}

		file_put_contents(LOG_DIR . DS . '_gallery_checkImageNames_ids.log', implode("\n", $arr));

		return $arr;
	}
}
