<?php declare(strict_types = 1);

namespace Gallery\Model\Entities;

use ArrayIterator;
use Core\Model\Entities\TId;
use Core\Model\Entities\TranslateListener;
use Core\Model\Entities\TTranslateListener;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\DBAL\Types\Types;
use Doctrine\ORM\Mapping as ORM;
use Exception;
use Nette\Utils\Strings;

/**
 * TODO odstranit texty v této entitě
 */
#[ORM\Table(name: 'gallery__album')]
#[ORM\Entity]
#[ORM\EntityListeners([AlbumListener::class, TranslateListener::class])]
class Album
{
	use TId;
	use TTranslateListener;

	#[ORM\Column(type: Types::STRING, length: 255, nullable: true)]
	public ?string $title = null;

	#[ORM\Column(type: Types::TEXT, nullable: true)]
	protected ?string $description = null;

	/**
	 * @var Collection<Image>
	 */
	#[ORM\OneToMany(targetEntity: Image::class, mappedBy: 'album', indexBy: 'id')]
	#[ORM\OrderBy(['position' => 'ASC'])]
	protected Collection $images;

	protected ?Image $cover = null;

	#[ORM\Column(type: Types::SMALLINT)]
	public int $isPublished = 0;

	/**
	 * @var Collection<AlbumText>
	 */
	#[ORM\OneToMany(targetEntity: AlbumText::class, mappedBy: 'album', indexBy: 'lang')]
	public Collection $texts;

	#[ORM\Column(type: Types::STRING, nullable: false)]
	public string $basePath;

	public function __construct(string $basePath)
	{
		$this->basePath    = $basePath;
		$this->isPublished = 1;
		$this->images      = new ArrayCollection;
		$this->texts       = new ArrayCollection;
	}

	public function __clone()
	{
		$this->id     = null;
		$this->images = new ArrayCollection;
		$this->texts  = new ArrayCollection;
	}

	public function setId(?int $id): void
	{
		$this->id = $id;
	}

	public function generatePath(): string
	{
		return $this->basePath . '/' . $this->getId() . ($this->title ? '-' . Strings::webalize($this->title) : '');
	}

	public function getPath(): string { return $this->basePath . '/album_' . $this->getId(); }

	public function setImages(array $images): void
	{
		$this->images = new ArrayCollection($images);
	}

	/**
	 * @return ArrayCollection<Image>
	 * @throws Exception
	 */
	public function getImages(): ArrayCollection
	{
		/** @var ArrayIterator $iterator */
		$iterator = $this->images->getIterator();
		$iterator->uasort(fn(Image $a, Image $b) => $a->getPosition() <=> $b->getPosition());

		return new ArrayCollection(iterator_to_array($iterator));
	}

	public function setCoverImage(?Image $img): void
	{
		$this->cover = $img;
	}

	public function getCoverRaw(): ?Image { return $this->cover; }

	public function getCoverImage(): ?Image
	{
		if (!$this->cover) {
			if ($this->images->count()) {
				foreach ($this->images->toArray() as $img) {
					if ($img->isCover) {
						$cover = $img;
						break;
					}
				}

				if (!isset($cover)) {
					$cover = $this->images->first();
				}

				$this->cover = $cover;
			}
		}

		return $this->cover ?: null;
	}

	public function getImagesCount(): int
	{
		return $this->images->count();
	}

	public function addImage(Image $image): void
	{
		$this->images->add($image);
	}

	/******
	 * === Texts
	 */

	public function addText(AlbumText $text): void
	{
		if (!$this->texts->containsKey($text->getLang())) {
			$this->texts->add($text);
		}
	}

	public function getText(?string $lang = null): ?AlbumText
	{
		return $this->texts->get($lang ?: $this->locale);
	}

	public function getTitle(): ?string { return $this->getText($this->locale)->title ?? $this->title; }

	public function getDescription(): ?string
	{
		return $this->getText($this->locale)->description ?? $this->description;
	}
}
