<?php declare(strict_types = 1);

namespace Gallery\AdminModule\Components\Image;

use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Images\ImageHelper;
use Core\Model\Lang\Lang;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Exception;
use Gallery\FrontModule\Model\Albums;
use Gallery\Model\Entities\Album;
use Gallery\Model\Entities\Image;
use Gallery\Model\Entities\ImageText;
use Gallery\Model\GalleryConfig;
use Gallery\Model\Images;
use InvalidArgumentException;
use Nette\Application\AbortException;
use Nette\Caching\Cache;
use Nette\ComponentModel\IComponent;
use Nette\Forms\Form;
use Nette\Http\FileUpload;
use Nette\Utils\ArrayHash;
use Nette\Utils\Arrays;
use Nette\Utils\FileSystem;
use Nette\Utils\Html;
use Nette\Utils\Json;
use Nette\Utils\Strings;
use Nette\Utils\Validators;
use Tags\Model\Tags;

/**
 * TODO popup náhled celé fotky fotky
 * TODO dodělat info o ajax stavu u každéh tlačítka
 * TODO handle requesty cele přes post (i ID) a refaktorovat response
 */
class ImagesZone extends BaseControl
{
	protected string              $uploadPath              = '';
	protected ?Album              $album                   = null;
	protected Tags                $tagsService;
	protected IImageFinderFactory $imageFinderFactory;
	public array                  $extendedResponse        = [];
	public array                  $extendedResponseOnEmpty = [];
	public bool                   $showImageFinder         = true;
	public array                  $data;
	protected Images              $imagesService;

	/** @var callable[] */
	public array $onBeforeUpload = [];

	/** @var callable[] */
	public array $onAfterUpload = [];

	/** @var callable[] */
	public array $onEmpty = [];

	/** @var callable[] */
	public array $onBeforeUpdate = [];

	/** @var callable[] */
	public array $onAfterUpdate = [];

	/** @var callable[] */
	public array $onAfterRemove = [];

	/** @var callable[] */
	public array $onAfterChangePosition = [];

	/** @var callable[] */
	public array $onPrepareDefaultImage = [];

	/** @var Lang[] */
	public array $contentLanguages = [];

	protected array $customFieldsLatte = [
		'before' => [],
		'after'  => [],
	];

	public const MODE_COPY = 0;
	public const MODE_MOVE = 1;

	public function __construct(
		Images              $images,
		Tags                $tags,
		IImageFinderFactory $imageFinderFactory
	)
	{
		$this->imagesService      = $images;
		$this->tagsService        = $tags;
		$this->imageFinderFactory = $imageFinderFactory;
	}

	public function render(): void
	{
		$this->template->showImageFinder   = $this->showImageFinder;
		$this->template->album             = $this->album;
		$this->template->imagesJson        = $this->prepareDefaultImages();
		$this->template->lang              = $this->translator->getLocale();
		$this->template->contentLanguages  = $this->contentLanguages;
		$this->template->customFieldsLatte = $this->customFieldsLatte;
		$this->template->render(__DIR__ . '/ImagesZone.latte');
	}

	/**
	 * @param IComponent $presenter
	 */
	protected function attached($presenter): void
	{
		parent::attached($presenter);
		$this->contentLanguages = $this->langsService->getLangs(false);
	}

	public function createComponentUploadFromServerForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setShowLangSwitcher(false)
			->unmonitor(Form::class);
		$form->addFilesManager('files', '')
			->setHtmlId('uploadFromServerForm__files')
			->multiple = true;
		$form->addHidden('albumId')
			->setHtmlAttribute('data-album-id')
			->setHtmlId('uploadFromServerForm__albumId');
		$form->addSubmit('submit', 'gallery.imageZone.submitFromServer', 'btn-primary')
			->setHtmlAttribute('type', 'button');

		$form->onValidate[] = [$this, 'formValidate'];
		$form->onSuccess[]  = [$this, 'formSuccess'];

		return $form;
	}

	public function formValidate(BaseForm $form, ArrayHash $values): void
	{
		if (empty($values->files)) {
			$form->addError('gallery.imageZone.errors.emptyFiles');
			$this->redrawControl('uploadFromServerForm');

			return;
		}
		if (!Arrays::every($this->parseFilePaths($values->files), static fn($path) => ImageHelper::isImage($path))) {
			$form->addError('gallery.imageZone.errors.notImage');
		}

		$this->redrawControl('uploadFromServerForm');
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): void
	{
		if ($form->getHttpData()['albumId'] === '') {
			$this->album = null;
			$this->onEmpty($this);
			$albumId = $this->album->getId();
		} else {
			$albumId = (int) $form->getHttpData()['albumId'];
		}

		$fileUploads = array_map(static fn($file) => new FileUpload([
			'name'     => basename($file),
			'size'     => filesize($file),
			'tmp_name' => $file,
			'error'    => 0,
		]), $this->parseFilePaths($values->files));

		$this->upload($albumId, $fileUploads, self::MODE_COPY);

		$this->clearCache($albumId);

		$this->presenter->payload->data += $this->extendedResponseOnEmpty;
		$this->presenter->sendPayload();
	}

	protected function clearCache(?int $albumId): void
	{
		if (!$albumId) {
			return;
		}

		$cache = new Cache($this->cacheStorage, Albums::CACHE_NAMESPACE);

		foreach ($this->translator->getLocalesWhitelist() as $l) {
			$cache->remove('album/' . $l . '/' . $albumId);
		}

		$cache->clean([Cache::Tags => ['album/' . $albumId]]);
	}

	/*******************************************************************************************************************
	 * =================  Handle
	 */

	public function handleGetTags(): void
	{
		$tags = [];
		foreach ($this->tagsService->getAvailableTags() as $tag) {
			$tags[$tag->getId()] = $tag->title;
		}

		$this->presenter->payload->data = $tags;
		$this->presenter->sendPayload();
	}

	public function handleTagsAutocomplete(string $q): void
	{
		$tags = [];
		foreach ($this->tagsService->getAvailableTags() as $tag) {
			$tags[] = ['key' => $tag->getId(), 'value' => $tag->title];
		}

		$this->presenter->sendJson($tags);
	}

	public function handleTagCreate(string $value): void
	{
		$this->presenter->sendJson($this->tagsService->createTag($value));
	}

	public function handleRemoveTag(string $value): void
	{
		$responseData = ['status' => 'error', 'errorMessage' => ''];
		$tag          = $this->tagsService->findByTitle($value);

		if ($tag) {
			$this->tagsService->remove($tag->getId());
			$responseData['status'] = 'ok';
		}

		$this->presenter->payload->data = $responseData;
		$this->presenter->sendPayload();
	}

	public function handleOnEmpty(): void
	{
		$responseData        = ['status' => 'error', 'errorMessage' => ''];
		$this->data['title'] = $this->presenter->getHttpRequest()->getPost('title');
		$this->onEmpty($this);

		if (!$this->album) {
			$responseData['errorMessage'] = $this->t('gallery.image.albumMissing');
		} else {
			$responseData['status']    = 'ok';
			$responseData['albumId']   = $this->album->getId();
			$responseData['albumPath'] = $this->album->generatePath();
		}

		$this->presenter->payload->data = $responseData + $this->extendedResponseOnEmpty;
		$this->presenter->sendPayload();
	}

	public function handleUpload(): void
	{
		$request     = $this->presenter->getHttpRequest();
		$fileUploads = $request->getFiles();
		$albumId     = (int) $request->getPost('albumId');

		$this->upload($albumId, $fileUploads);

		$this->clearCache($albumId);

		$this->presenter->sendPayload();
	}

	/**
	 * @param int|string $id
	 *
	 * @throws AbortException
	 */
	public function handleUpdateImage($id): void
	{
		$responseData = ['status' => 'error', 'errorMessage' => ''];
		$request      = $this->presenter->getHttpRequest();

		$position = $request->getPost('position');
		if (is_numeric($position)) {
			$this->changePosition((int) $id, (int) $position);

			return;
		}

		/** @var Image|null $image */
		$image = $this->em->getRepository(Image::class)->find($id);
		$error = false;

		if (!$image) {
			$error = true;
		}

		if (!$error) {
			$data = $request->getPost('data');

			if (!$data) {
				$data[] = [
					'key'   => $request->getPost('key'),
					'value' => $request->getPost('value'),
					'lang'  => $request->getPost('lang'),
				];
			}

			$this->em->beginTransaction();
			try {
				foreach ($data as $v) {
					$this->onBeforeUpdate($id, $v);
					$key   = $v['key'];
					$value = $v['value'];
					$lang  = $v['lang'];

					if ($key && $value !== null && property_exists(Image::class, $key)) {
						if ($key == 'isCover' && $value == 1) {
							$this->imagesService->setAsCover($image->getId());
						} else if ($key == 'tags') {
							$tags = $value == '' ? [] : array_map(function($id) {
								return $this->tagsService->getReference($id);
							}, explode(',', $value));
							$image->setTags($tags);
						} else if ($lang) {
							$txt = $image->getText($lang);
							if (!$txt) {
								$txt = new ImageText($image, $lang);
								$image->addText($txt);
							}
							if (GalleryConfig::isAttrAllow($key)) {
								$txt->$key = $value;
							}
							$this->em->persist($txt);
						} else {
							$image->$key = $value;
						}

						$this->em->persist($image);
						$this->onAfterUpdate($image);

						$responseData[$key]['status'] = 'ok';
					}
				}
				$this->em->flush();
				$this->em->commit();

				if ($image->getAlbum()) {
					$this->clearCache($image->getAlbum()->getId());
				}

				$responseData['status'] = 'ok';
			} catch (Exception $e) {
				$responseData['status'] = 'error';
				$this->em->rollback();
			}
		}

		$this->presenter->payload->data = $responseData + $this->extendedResponse;
		$this->presenter->sendPayload();
	}

	public function changePosition(int $id, int $position): void
	{
		$responseData = ['status' => 'error', 'errorMessage' => ''];
		/** @var Image|null $image */
		$image = $this->em->getRepository(Image::class)->find($id);

		if ($image) {
			$image->setPosition($position);
			$this->em->persist($image);
			$this->em->flush();

			$responseData['status'] = 'ok';
		}

		$this->onAfterChangePosition($image);

		$this->presenter->payload->data = $responseData + $this->extendedResponse;

		if ($image && $image->getAlbum()) {
			$this->clearCache($image->getAlbum()->getId());
		}

		$this->presenter->sendPayload();
	}

	public function handleRemoveImage(): void
	{
		$responseData = ['status' => 'error', 'errorMessage' => ''];
		$request      = $this->presenter->getHttpRequest();
		$id           = (int) $request->getPost('id');

		if ($this->imagesService->removeImage($id)) {
			$responseData['status'] = 'ok';
		}

		$this->onAfterRemove($id);

		$this->presenter->payload->data = $responseData + $this->extendedResponse;
		$this->presenter->sendPayload();
	}

	public function handleUseFoundImage(): void
	{
		$presenter = $this->presenter;
		try {
			$imgId   = $presenter->request->getPost('imgId');
			$albumId = $presenter->request->getPost('albumId');

			if (!$imgId || !$albumId) {
				throw new InvalidArgumentException();
			}

			$img = $this->imagesService->get($imgId);

			if ($img) {
				/** @var Album $album */
				$album = $this->em->getReference(Album::class, $albumId);

				$image = clone $img;
				$image->setAlbum($album);
				$image->setCloneOf($img);
				$image->clear();
				$img->setLastUse();

				$this->em->persist($img);
				$this->em->persist($image);

				foreach ($img->getTexts()->toArray() as $t) {
					/** @var ImageText $t */
					$it              = new ImageText($image, $t->getLang());
					$it->title       = $t->title;
					$it->alt         = $t->alt;
					$it->description = $t->description;
					$it->source      = $t->source;

					$this->em->persist($it);
				}

				$this->em->flush();

				$data = [
					'id'          => $image->getId(),
					'filename'    => $image->getFilename(),
					'dir'         => $image->path,
					'size'        => $image->getSize(),
					'source'      => [],
					'title'       => [],
					'alt'         => [],
					'description' => [],
					'link'        => $image->link,
					'tags'        => [],
					'isPublished' => $image->isPublished,
					'isCover'     => $image->isCover,
					'thumbCoords' => $image->getThumbCoords(),
				];

				foreach ($this->contentLanguages as $l) {
					$data['source'][$l->getTag()]      = $image->getText($l->getTag())->source;
					$data['title'][$l->getTag()]       = $image->getText($l->getTag())->title;
					$data['description'][$l->getTag()] = $image->getText($l->getTag())->description;
					$data['alt'][$l->getTag()]         = $image->getText($l->getTag())->alt;
				}

				$presenter->payload->image = $data;

				$presenter->flashMessageSuccess('gallery.imageZone.imageAdded');
			}
		} catch (Exception $e) {
			$presenter->flashMessageDanger('gallery.imageZone.imageAddFailed');
			throw $e;
		}

		$presenter->redrawControl('flashes');
	}

	/*******************************************************************************************************************
	 * =================  Get/Set
	 */

	private function prepareDefaultImages(): string
	{
		$data = [];

		if ($this->album) {
			foreach ($this->album->getImages() as $image) {
				/** @var Image $image */
				$arr = [
					'id'          => $image->getId(),
					'filename'    => $image->getFilename(),
					'dir'         => $image->path,
					'size'        => $image->getSize(),
					'source'      => [],
					'title'       => [],
					'alt'         => [],
					'description' => [],
					'link'        => $image->link,
					'tags'        => array_map(static function($arr) {
						return ['key' => $arr->getId(), 'value' => $arr->title];
					}, $image->getTags()->toArray()),
					'isPublished' => $image->isPublished,
					'isCover'     => $image->isCover,
					'thumbCoords' => $image->getThumbCoords(),
				];

				foreach ($this->contentLanguages as $l) {
					$arr['source'][$l->getTag()]      = $image->getText($l->getTag())->source;
					$arr['title'][$l->getTag()]       = $image->getText($l->getTag())->title;
					$arr['description'][$l->getTag()] = $image->getText($l->getTag())->description;
					$arr['link'][$l->getTag()]        = $image->getText($l->getTag())->link;
					$arr['alt'][$l->getTag()]         = $image->getText($l->getTag())->alt;
				}

				foreach ($this->onPrepareDefaultImage as $c) {
					$c($image, $arr);
				}

				$data[] = $arr;
			}
		}

		return Json::encode($data);
	}

	public function setUploadPath(string $path): self
	{
		if (!Validators::isNone($path)) {
			$this->uploadPath = $path;
		}

		return $this;
	}

	public function getUploadPath(): string { return $this->uploadPath; }

	public function getUploadDir(): string { return WWW_DIR . '/' . $this->getUploadPath(); }

	/**
	 * @param int|string $id
	 *
	 * @throws Exception
	 */
	public function setAlbum($id): void
	{
		$this->album = $this->em->getRepository(Album::class)->createQueryBuilder('a')->where('a.id = :id')->setParameter('id', $id)
			->leftJoin('a.images', 'imgs')->leftJoin('imgs.tags', 'tags')
			->addSelect('imgs, tags')->getQuery()->getOneOrNullResult();

		if (!$this->album) {
			if ($this->getPresenter(false)) {
				$this->getPresenter(false)->error();
			}
		} else {
			$this->setUploadPath($this->album->generatePath());
			if ($this->showImageFinder) {
				$this['imageFinder']->setActiveAlbum($this->album);
			}

			$this['uploadFromServerForm']->getComponent('albumId')->setValue($this->album->getId());
			$this->redrawControl('uploadFromServerForm-snippet');
		}
	}

	public function getAlbum(): ?Album { return $this->album; }

	public function getEm(): EntityManagerDecorator
	{
		return $this->em;
	}

	/**
	 * @return string[]
	 */
	protected function parseFilePaths(string $input): array
	{
		$sep = '|';
		if (Strings::contains($input, $sep)) {
			return array_map(static fn($s) => WWW_DIR . DS . $s, array_map('trim', explode($sep, $input)));
		}

		return [WWW_DIR . DS . $input];
	}

	public function addCustomFieldLatteBefore(string $file): self
	{
		$this->customFieldsLatte['before'][] = $file;

		return $this;
	}

	public function addCustomFieldLatteAfter(string $file): self
	{
		$this->customFieldsLatte['after'][] = $file;

		return $this;
	}

	/*******************************************************************************************************************
	 * =================  Components
	 */

	protected function createComponentImageFinder(): ImageFinder
	{
		return $this->imageFinderFactory->create();
	}

	/**
	 * @param int|string $albumId
	 *
	 * @throws Exception
	 */
	public function handleShowGallery($albumId = null): void
	{
		if ($albumId) {
			$this->setAlbum($albumId);
		}
		$this->presenter->getTemplate()->modal            = 'test';
		$this->presenter->getTemplate()->modalDialogClass = 'modal-full';
		ob_start();
		$this->render();
		$body                                      = ob_get_clean();
		$this->presenter->getTemplate()->modalBody = Html::el()->addHtml($body);
		$this->presenter->redrawControl('modal');
	}

	/**
	 * @param int          $albumId
	 * @param FileUpload[] $fileUploads
	 * @param int          $mode
	 */
	protected function upload(int $albumId, array $fileUploads, int $mode = self::MODE_MOVE): void
	{
		$responseData = ['status' => 'error', 'errorMessage' => ''];
		$error        = false;

		$this->onBeforeUpload($this);

		try {
			$this->setAlbum($albumId);
		} catch (Exception $e) {
			$error                        = true;
			$responseData['errorMessage'] = $this->t('gallery.image.albumMissing');
		}

		if (Validators::isNone($this->getUploadPath())) {
			$error                        = true;
			$responseData['errorMessage'] = $this->t('gallery.image.uploadPathMissing');
		}

		if (!$error) {
			foreach ($fileUploads as $file) {
				if ($file->isOk()) {
					$this->em->beginTransaction();
					try {
						$filename = $file->getSanitizedName();
						$image    = new Image($this->album, $filename);

						$this->em->persist($image);
						$this->em->flush();

						$filename        = $image->getId() . '_' . $filename;
						$image->filename = $filename;
						$this->em->persist($image);
						$this->em->flush();

						if ($mode === self::MODE_MOVE) {
							$file->move($this->getUploadDir() . '/' . $filename);
						} else {
							FileSystem::copy($file->getTemporaryFile(), $this->getUploadDir() . '/' . $filename);
						}

						ImageHelper::autoResize($image->getFile());
						$this->onAfterUpload($image);

						$uploadStatus = 'ok';
						$data         = [
							'albumPath'   => $this->album->generatePath(),
							'id'          => $image->getId(),
							'filename'    => $image->getFilename(),
							'dir'         => $this->album->generatePath(),
							'size'        => $image->getSize(),
							'link'        => $image->link,
							'source'      => [],
							'title'       => [],
							'alt'         => [],
							'description' => [],
							'tags'        => [],
							'isPublished' => $image->isPublished,
							'isCover'     => $image->isCover,
							'thumbCoords' => $image->getThumbCoords(),
						];

						foreach ($this->contentLanguages as $l) {
							$data['source'][$l->getTag()]      = $image->getText($l->getTag())->source;
							$data['title'][$l->getTag()]       = $image->getText($l->getTag())->title;
							$data['description'][$l->getTag()] = $image->getText($l->getTag())->description;
							$data['alt'][$l->getTag()]         = $image->getText($l->getTag())->alt;
						}

						$this->em->commit();
					} catch (Exception $e) {
						$this->em->rollback();
						@unlink($file->getTemporaryFile());
						@unlink($this->getUploadDir() . '/' . $filename);

						$uploadStatus = 'error';
						$data         = [];
					}
				} else {
					$uploadStatus = 'error';
					$data         = [];
				}

				$responseData['files'][$file->getName()] = [
					'status' => $uploadStatus,
					'data'   => $data,
				];

				$responseData['file'] = [
					'status' => $uploadStatus,
					'data'   => $data,
				];
			}

			if ($this->album) {
				$responseData['albumId'] = $this->album->getId();
			}

			$responseData['status'] = 'ok';
		}

		$this->presenter->payload->data = $responseData + $this->extendedResponse;
	}
}
