<?php declare(strict_types = 1);

namespace Gallery\Model\Entities;

use Core\Model\Entities\TTranslateListener;
use Core\Model\Providers\ISiteMapImage;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use Exception;
use Gedmo\Mapping\Annotation as Gedmo;
use Core\Model\Entities\TId;
use Nette\Utils\DateTime;
use Nette\Utils\Json;
use Tags\Model\Entities\Tag;

/**
 * @ORM\Table(name="gallery__image")
 * @ORM\Entity
 * @ORM\EntityListeners({"Core\Model\Entities\TranslateListener", "ImageListener"})
 */
class Image implements ISiteMapImage
{
	use TId;
	use TTranslateListener;

	/**
	 * @ORM\Column(type="string", length=255, nullable=true)
	 */
	protected ?string $source = null;

	/**
	 * @ORM\Column(name="title", type="text", nullable=true)
	 */
	protected ?string $title = null;

	/**
	 * @ORM\Column(type="text", nullable=true)
	 */
	public ?string $description = null;

	/**
	 * @ORM\Column(type="text", nullable=true)
	 */
	public ?string $alt = null;

	/**
	 * @ORM\Column(type="text", nullable=true)
	 */
	public ?string $link = null;

	/**
	 * @ORM\Column(type="string", nullable=false)
	 */
	public string $filename;

	/**
	 * @ORM\Column(type="string", nullable=true)
	 */
	public ?string $path = null;

	/**
	 * TODO manyToMany -> nebude potřeba cloneOf. Budou potřeba úpravy i v komponentě
	 *
	 * @Gedmo\SortableGroup
	 * @ORM\ManyToOne(targetEntity="Album", inversedBy="images")
	 * @ORM\JoinColumn(name="album_id", referencedColumnName="id", onDelete="CASCADE")
	 */
	protected ?Album $album;

	/**
	 * @var Collection<Tag>
	 * @ORM\ManyToMany(targetEntity="Tags\Model\Entities\Tag")
	 * @ORM\JoinTable(name="gallery__image_tags",
	 *     joinColumns={@ORM\JoinColumn(name="image_id", referencedColumnName="id", onDelete="CASCADE")},
	 *     inverseJoinColumns={@ORM\JoinColumn(name="tag_id", referencedColumnName="id", onDelete="CASCADE")}
	 *     )
	 */
	protected Collection $tags;

	/**
	 * @Gedmo\SortablePosition
	 * @ORM\Column(type="integer")
	 */
	private int $position;

	/**
	 * @var int
	 * @ORM\Column(type="smallint")
	 */
	public $isPublished;

	/**
	 * @var int
	 * @ORM\Column(type="smallint")
	 */
	public $isCover;

	/**
	 * @var array|string|null
	 * @ORM\Column(type="array", nullable=true)
	 */
	public $thumbCoords;

	/**
	 * @ORM\ManyToOne(targetEntity="Image")
	 * @ORM\JoinColumn(name="clone_of_id", referencedColumnName="id", onDelete="SET NULL", nullable=true)
	 *
	 */
	protected ?Image $cloneOf = null;

	/**
	 * @var DateTime|null
	 * @ORM\Column(type="datetime", nullable=true)
	 */
	private $lastUse;

	/**
	 * @var Collection<ImageText>
	 * @ORM\OneToMany(targetEntity="ImageText", mappedBy="image", indexBy="lang", cascade={"all"})
	 */
	protected Collection $texts;

	public function __construct(?Album $album, string $filename)
	{
		$this->album       = $album;
		$this->filename    = $filename;
		$this->isPublished = 1;
		$this->isCover     = 0;
		$this->tags        = new ArrayCollection;

		if ($album) {
			$this->path = $album->generatePath();
		}

		$this->texts = new ArrayCollection;
		$this->setPosition(-1);
		$this->lastUse = new DateTime;
	}

	public function clear(): void
	{
		$this->source      = null;
		$this->description = null;
		$this->link        = null;
		$this->isPublished = 1;
		$this->isCover     = 0;
		$this->setTags([]);
		$this->setPosition(-1);
		$this->texts = new ArrayCollection;
	}

	public function getFilename(): string { return $this->filename; }

	public function getFilePath(): string { return $this->path . '/' . $this->getFilename(); }

	public function getFile(): string { return WWW_DIR . $this->getFilePath(); }

	public function getSize(): int
	{
		$file = $this->getFile();

		if (file_exists($file)) {
			return filesize($file);
		}

		return 0;
	}

	/******
	 * === Album
	 */

	public function getAlbum(): ?Album { return $this->album; }

	public function setAlbum(Album $album): void
	{
		$this->album   = $album;
		$this->lastUse = new DateTime;
	}

	/******
	 * === Clone of
	 */

	public function getCloneOf(): ?Image { return $this->cloneOf; }

	public function setCloneOf(Image $image): void
	{
		$this->cloneOf = $image;
		$this->lastUse = new DateTime();
	}

	/******
	 * === Thumb coords
	 */

	public function getThumbCoords(): ?array
	{
		if ($this->thumbCoords === null) {
			return null;
		}

		try {
			if (is_string($this->thumbCoords)) {
				$this->thumbCoords = $this->thumbCoords ? Json::decode($this->thumbCoords, Json::FORCE_ARRAY) : [];
			}

			return $this->thumbCoords;
		} catch (Exception $e) {
		}

		return null;
	}

	/******
	 * === Last use
	 */

	public function setLastUse(): void { $this->lastUse = new DateTime; }

	/**
	 * @return DateTime|null
	 */
	public function getLastUse() { return $this->lastUse; }

	/******
	 * === Tags
	 */

	public function addTag(Tag $tag): void
	{
		if ($this->tags->contains($tag)) {
			return;
		}

		$this->tags->add($tag);
	}

	public function removeTag(Tag $tag): void
	{
		if (!$this->tags->contains($tag)) {
			return;
		}

		$this->tags->removeElement($tag);
	}

	/**
	 * @param Tag[] $tags
	 */
	public function setTags(array $tags): void
	{
		$this->tags = new ArrayCollection($tags);
	}

	/**
	 * @return Collection<Tag>
	 */
	public function getTags(): Collection { return $this->tags; }

	/**
	 * @return string[]
	 */
	public function getTagsTitles(): array
	{
		$tags = [];
		foreach ($this->getTags() as $tag) {
			$tags[$tag->getId()] = $tag->title;
		}

		return $tags;
	}

	/******
	 * === Position
	 */

	public function setPosition(int $position): void
	{
		$this->position = $position;
	}

	public function getPosition(): int
	{
		return $this->position;
	}

	public function __clone()
	{
		$this->id      = null;
		$this->lastUse = new DateTime;
		$this->texts   = new ArrayCollection;
	}

	/******
	 * === Texts
	 */

	public function addText(ImageText $text): void
	{
		if (!$this->texts->containsKey($text->getLang())) {
			$this->texts->add($text);
		}
	}

	/**
	 * @return Collection<ImageText>
	 */
	public function getTexts() { return $this->texts; }

	/**
	 * @param ImageText[] $texts
	 */
	public function setTexts(array $texts): void
	{
		$this->texts = new ArrayCollection($texts);
	}

	public function getText(string $lang = null): ?ImageText
	{
		return $this->texts->get($lang ?: $this->locale);
	}

	public function getTitle(): ?string { return $this->getText($this->locale)->title ?? $this->title; }

	public function getDescription(): ?string { return $this->getText($this->locale)->description ?? $this->description; }

	public function getSource(): ?string { return $this->getText($this->locale)->source ?? $this->source; }

	/******
	 * === Dodatečné ISiteMapImage
	 */

	public function getCaption(): ?string { return $this->description; }

	public function getGeoLocation(): string { return ''; }

	public function getLicense(): string { return ''; }

}
