<?php declare(strict_types = 1);

namespace Gallery\Model;

use Core\Model\Helpers\BaseEntityService;
use Core\Model\Images\ImagePipe;
use Doctrine\ORM\NonUniqueResultException;
use Exception;
use Gallery\Model\Entities\Image;
use Nette\Utils\FileSystem;

/**
 * @method Image|null getReference($id)
 * @method Image[]|null getAll()
 * @method Image|null get($id)
 */
class Images extends BaseEntityService
{
	protected $entityClass = Image::class;

	protected ImagePipe $imagePipe;

	public function __construct(ImagePipe $imagePipe)
	{
		$this->imagePipe = $imagePipe;
	}

	/**
	 * @param int|string|int[]|string[] $albumId
	 *
	 * @return Image[]
	 */
	public function getPublishedByAlbumId($albumId): array
	{
		return $this->getEr()->createQueryBuilder('i')
			->andWhere(is_array($albumId) ? 'i.album IN (:aId)' : 'a.album = :aId')->setParameter('aId', $albumId)
			->getQuery()->getResult();
	}

	/**
	 * @return Image[]
	 */
	public function getImages(): array { return $this->getEr()->findBy([], ['position' => 'ASC']); }

	/**
	 * @param int|string $id
	 */
	public function getImage($id): ?Image { return $this->getEr()->find($id); }

	/**
	 * @param int|string $albumId
	 *
	 * @throws NonUniqueResultException
	 */
	public function getCoverByAlbum($albumId): ?Image
	{
		return $this->getEr()->createQueryBuilder('i')->andWhere('i.isCover = 1 OR (i.isPublished = 1 AND i.isCover = 0)')
			->andWhere('i.album = :album')->setParameter('album', $albumId)->orderBy('i.position', 'ASC')
			->getQuery()->setMaxResults(1)->getOneOrNullResult();
	}

	/**
	 * @param string|int $id
	 */
	public function setAsCover($id): bool
	{
		if ($image = $this->getEr()->find($id)) {
			foreach ($this->getEr()->findBy(['isCover' => 1, 'album' => $image->getAlbum()->getId()]) as $img) {
				$img->isCover = 0;
				$this->em->persist($img);
			}

			$image->isCover = 1;
			$this->em->persist($image);

			$this->em->flush();

			return true;
		}

		return false;
	}

	/**
	 * @param string|int $id
	 */
	public function remove($id): bool
	{
		if ($image = $this->getEr()->find($id)) {
			/** @var Image $image */
			$same = $this->getEr()->createQueryBuilder('i')
				->where('i.filename = :filename')
				->andWhere('i.path = :path')
				->andWhere('i.id != :id')
				->setParameters([
					'filename' => $image->getFilename(),
					'path'     => $image->path,
					'id'       => $id,
				])->getQuery()->getArrayResult();

			if (empty($same) && file_exists($image->getFile())) {
				FileSystem::delete($image->getFile());
			}

			$this->imagePipe->removeThumbs($image->getFilePath());
			$this->em->remove($image);
			$this->em->flush();

			return true;
		}

		return false;
	}

	public function search(string $q): array
	{
		$values = array_map('trim', explode(',', $q));

		$qr = $this->getEr()->createQueryBuilder('i');

		foreach ($values as $k => $v) {
			$qr->orWhere('i.description LIKE :k' . $k)->orWhere('i.source LIKE :k' . $k)->orWhere('i.filename LIKE :k' . $k)
				->orWhere('i.title LIKE :k' . $k)->orWhere('i.path LIKE :k' . $k)
				->leftJoin('i.tags', 'tags')->orWhere('tags.title LIKE :k' . $k)
				->leftJoin('i.album', 'a')->addSelect('a')
				->setParameter('k' . $k, "%$v%");
		}

		$images = [];
		foreach ($qr->getQuery()->getResult() as $image) {
			/** @var Image $image */
			$key = $image->getCloneOf() !== null ? $image->getCloneOf()->getId() : $image->getId();

			if (isset($images[$key]) && $image->getLastUse() && $images[$key]->getLastUse() && $image->getLastUse()->getTimestamp() < $images[$key]->getLastUse()->getTimestamp()) {
				continue;
			}

			$images[$key] = $image;
		}

		usort($images, static function(Image $a, Image $b) {
			$aLast = $a->getLastUse() ? $a->getLastUse()->getTimestamp() : null;
			$bLast = $b->getLastUse() ? $b->getLastUse()->getTimestamp() : null;

			return $aLast <=> $bLast;
		});

		return array_reverse($images);
	}

	/**
	 * @param int|string $id
	 *
	 * @return Image[]
	 */
	public function getByAlbum($id): array
	{
		return $this->getEr()->findBy(['album' => $id]);
	}

	public function removeImage(int $id): bool
	{
		return $this->remove($id);
	}
}
