<?php declare(strict_types = 1);

namespace Gallery\FrontModule\Model;

use Core\Model\Helpers\BaseFrontEntityService;
use Gallery\FrontModule\Model\Dao;
use Gallery\Model\Entities\Album;
use Nette\Caching\Cache;
use Nette\DI\Attributes\Inject;

class Albums extends BaseFrontEntityService
{
	final public const CACHE_NAMESPACE = 'albums';

	#[Inject]
	public Images $imagesService;

	protected       $entityClass = Album::class;
	protected array $cAlbums     = [];
	protected array $cacheDep    = [
		Cache::Tags   => ['albums'],
		Cache::Expire => '1 week',
	];

	public function getCache(): Cache
	{
		if ($this->cache === null) {
			$this->cache = new Cache($this->cacheStorage, self::CACHE_NAMESPACE);
		}

		return $this->cache;
	}

	/**
	 * @return Dao\Album[]
	 */
	public function getAlbums(array $ids): array
	{
		$whereIds = [];
		$result   = [];
		$locale   = $this->translator->getLocale();

		$keys = [];
		foreach ($ids as $aId) {
			if (isset($this->cAlbums[$aId])) {
				$result[$aId] = $this->cAlbums[$aId];
			} else {
				$keys[] = 'album/' . $locale . '/' . $aId;
			}
		}

		if (!empty($keys)) {
			foreach ($this->getCache()->bulkLoad($keys) as $key => $album) {
				$tmp = explode('/', $key);
				$id  = end($tmp);

				if ($album) {
					$result[$id]        = $album;
					$this->cAlbums[$id] = $album;
				} else {
					$whereIds[]  = $id;
					$result[$id] = null;
				}
			}
		}

		$loadImages = [];
		if (!empty($whereIds)) {
			foreach ($this->getEr()->createQueryBuilder('a', 'a.id')
				         ->addSelect('at')
				         ->andWhere('a.id IN (:ids)')
				         ->leftJoin('a.texts', 'at', 'WITH', 'at.lang = :lang')
				         ->andWhere('a.isPublished = 1')
				         ->groupBy('a.id')
				         ->setParameters([
					         'ids'  => $whereIds,
					         'lang' => $locale,
				         ])
				         ->getQuery()->getArrayResult() as $row) {
				$dao                     = $this->fillDao($row);
				$result[$dao->id]        = $dao;
				$this->cAlbums[$dao->id] = $dao;
				$loadImages[$dao->id]    = &$result[$dao->id];
			}
		}

		if (!empty($loadImages)) {
			$this->setImages($loadImages);

			foreach ($loadImages as $album) {
				$cacheDep                = $this->cacheDep;
				$cacheDep[Cache::Tags][] = 'album/' . $album->id;
				$this->getCache()
					->save('album/' . $this->translator->getLocale() . '/' . $album->id, $album, $cacheDep);
			}
		}

		return $result;
	}

	public function get(int $id): ?Dao\Album
	{
		return $this->getAlbums([$id])[$id] ?? null;
	}

	protected function fillDao(array $a): Dao\Album
	{
		$locale = $this->translator->getLocale();

		$album              = new Dao\Album;
		$album->id          = $a['id'];
		$album->title       = $a['texts'][$locale]['title'];
		$album->description = $a['texts'][$locale]['description'];

		return $album;
	}

	/**
	 * @param Dao\Album[] $albums
	 */
	protected function setImages(array &$albums): void
	{
		$albumIds = [];
		foreach ($albums as $album) {
			$albumIds[] = $album->id;
		}

		$allImages = $this->imagesService->getImagesByAlbum($albumIds);

		foreach ($albums as &$album) {
			$cover = null;
			$imgs  = $allImages[$album->id] ?? [];

			foreach ($imgs as $k => $i) {
				if ($i->isCover) {
					$cover = $i;
					unset($imgs[$k]);
					break;
				}
			}

			if (!$cover && $imgs) {
				/** @var Dao\Image $cover */
				$cover          = reset($imgs);
				$cover->isCover = 1;
			}

			if ($cover) {
				$album->setCover($cover);
				$album->defaultCover = $cover;
			}

			$album->setImages(array_merge([$cover], $imgs));
		}
	}

}
