<?php declare(strict_types = 1);

namespace Gallery\AdminModule\Model;

use Core\Model\Helpers\BaseEntityService;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\NonUniqueResultException;
use Exception;
use Gallery\Model\Entities\Album;
use Gallery\Model\Entities\AlbumText;
use Gallery\Model\Entities\Image;
use Gallery\Model\Entities\ImageText;
use Nette\Utils\FileSystem;

/**
 * @method Album|null|object getReference($id)
 * @method Album[]|null getAll()
 * @method Album|null get($id)
 */
class Albums extends BaseEntityService
{
	public const CACHE_NAMESPACE = 'albums';

	protected $entityClass = Album::class;

	public function getCount(): int { return $this->getEr()->count([]); }

	/**
	 * @return Album[]
	 * @throws NonUniqueResultException
	 */
	public function getAlbums(?int $limit = null, ?int $offset = null): array
	{
		$qb = $this->getEr()->createQueryBuilder('a')->select('a.id')->orderBy('a.id', 'DESC');

		if ($limit !== null) {
			$qb->setMaxResults($limit);
		}
		if ($offset !== null) {
			$qb->setFirstResult($offset);
		}

		$result = $qb->getQuery()->getResult();
		$ids    = array_map(static function($a) { return $a['id']; }, $result);

		$albums = [];
		foreach ($ids as $id) {
			$albums[$id] = $this->getAlbum($id);
		}

		return $albums;
	}

	/**
	 * @param string|int $id
	 *
	 * @throws NonUniqueResultException
	 */
	public function getAlbum($id): ?Album
	{
		$qb = $this->getEr()->createQueryBuilder('a')
			->addSelect('at, i, it')
			->leftJoin('a.texts', 'at')
			->leftJoin('a.images', 'i')
			->leftJoin('i.texts', 'it')
			->where('a.id = :album')->setParameter('album', $id);

		return $qb->getQuery()->getOneOrNullResult();
	}

	public function deleteAlbum(int $id): bool
	{
		/** @var Album|null $album */
		$album = $this->getEr()->find($id);
		if ($album) {
			if ($album->getImagesCount() !== 0) {
				throw new Exception('gallery.album.cannotDeleteContainsImages');
			}

			$this->em->remove($album);
			$this->em->flush();

			return true;
		}

		return false;
	}

	public function cloneAlbum(Album $album): Album
	{
		$newAlbum = clone $album;

		$newTexts = [];
		foreach ($album->texts->toArray() as $text) {
			/** @var AlbumText $newText */
			$newText = clone $text;
			$newText->setAlbum($newAlbum);

			$newTexts[] = $newText;
			$this->em->persist($newText);
		}
		$newAlbum->texts = new ArrayCollection($newTexts);

		$newImages = [];
		foreach ($album->getImages()->toArray() as $img) {
			/** @var Image $newImg */
			$newImg = clone $img;
			$newImg->setAlbum($newAlbum);

			$newImgTexts = [];
			foreach ($img->getTexts()->toArray() as $imgText) {
				/** @var ImageText $newImgText */
				$newImgText = clone $imgText;
				$newImgText->setImage($newImg);
				$this->em->persist($newImgText);

				$newImgTexts[] = $newImgText;
			}
			$newImg->setTexts($newImgTexts);

			$this->em->persist($newImg);
			$newImages[] = $newImg;
		}
		$newAlbum->setImages($newImages);

		if (!empty($newImages)) {
			$cover = null;
			foreach ($newImages as $img) {
				if ($img->isCover) {
					$cover = $img;
					break;
				}
			}
			$newAlbum->setCoverImage($cover ?: $newImages[0]);
		}

		$this->em->persist($newAlbum);
		$this->em->flush();

		return $newAlbum;
	}

	public function removeAlbum(Album $album): bool
	{
		$paths  = [];
		$exists = [];

		foreach ($album->getImages() as $img) {
			$paths[$img->path][$img->filename] = $img->filename;
		}

		foreach ($paths as $path => $names) {
			foreach ($this->em->getRepository(Image::class)->createQueryBuilder('i')
				         ->select('i.id, i.filename, i.path')
				         ->where('i.path = :path')
				         ->andWhere('i.filename IN (\'' . implode("', '", $names) . '\')')
				         ->setParameters([
					         'path' => $path,
				         ])
				         ->getQuery()->getArrayResult() as $row) {
				$exists[$row['path']][$row['filename']][$row['id']] = $row['id'];
			}
		}

		foreach ($album->getImages() as $img) {
			$exist = $exists[$img->path][$img->filename] ?? [];

			if (count($exist) <= 1) {
				FileSystem::delete($img->getFile());
			}

			$this->em->remove($img);
		}

		$this->em->remove($album);

		return true;
	}
}
