<?php declare(strict_types = 1);

namespace Gallery\FrontModule\Model;

use Core\Model\Helpers\Arrays;
use Core\Model\Helpers\BaseFrontEntityService;
use Gallery\Model\Entities\Image;
use Gallery\FrontModule\Model\Dao;
use Nette\Utils\Json;

class Images extends BaseFrontEntityService
{
	public const CACHE_NAMESPACE = 'images';

	protected $entityClass = Image::class;

	/** @var Dao\Image[][] */
	protected array $cImages = [];

	public function getCoverImage(int $albumId): ?Dao\Image
	{
		if (!$albumId) {
			return null;
		}

		$imgs = $this->getImagesByAlbum($albumId);
		foreach ($imgs as $img) {
			if ($img->isCover) {
				return $img;
			}
		}

		return $imgs[0] ?? null;
	}

	/**
	 * @param int|int[] $albumId
	 *
	 * @return Dao\Image[]|array<int, Dao\Image[]>
	 */
	public function getImagesByAlbum($albumId): array
	{
		$oAlbumId = $albumId;
		$onlyOne  = !is_array($albumId);
		$loaded   = [];

		if (!is_array($albumId)) {
			$albumId = [$albumId];
		}

		foreach ($albumId as $k => $id) {
			if (isset($this->cImages[$id])) {
				$loaded[$k] = $this->cImages[$id];
			}
		}

		if (!empty($albumId)) {
			$raw = [];
			foreach ($this->getEr()->createQueryBuilder('i', 'i.id')
				         ->addSelect('it, IDENTITY(i.album) as alb')
				         ->where('i.isPublished = 1')
				         ->andWhere('i.album IN (:album)')->setParameter('album', array_values($albumId))
				         ->leftJoin('i.texts', 'it', 'WITH', 'it.lang = :lang')
				         ->setParameter('lang', $this->translator->getLocale())
				         ->orderBy('i.position')
				         ->groupBy('i.id')
				         ->getQuery()->getArrayResult() as $row) {
				$raw[$row['alb']][$row[0]['id']] = $row[0];
			}

			foreach ($raw as $id => $images) {
				$this->cImages[$id] = $this->fillDao($images);
			}

			foreach ($albumId as $id) {
				$loaded[$id] = $this->cImages[$id];
			}
		}

		if ($onlyOne) {
			return $loaded[$oAlbumId] ?? [];
		}

		return $loaded;
	}

	/**
	 * @return Dao\Image[]
	 */
	protected function fillDao(array $images): array
	{
		$imgs   = [];
		$locale = $this->translator->getLocale();

		foreach ($images as $img) {
			$i              = new Dao\Image;
			$i->id          = $img['id'];
			$i->title       = $img['texts'][$locale]['title'];
			$i->alt         = $img['texts'][$locale]['alt'];
			$i->description = $img['texts'][$locale]['description'];
			$i->source      = $img['texts'][$locale]['source'];
			$i->link        = $img['texts'][$locale]['link'];
			$i->position    = $img['position'];
			$i->isCover     = $img['isCover'];
			$i->albumId     = $img['alb'] ? (int) $img['alb'] : null;
			$i->setFile($img['path'] . DIRECTORY_SEPARATOR . $img['filename']);

			if ($img['thumbCoords']) {
				if (is_array($img['thumbCoords'])) {
					$i->thumbCoords = $img['thumbCoords'];
				} else if (Arrays::isJson($img['thumbCoords'])) {
					$i->thumbCoords = Json::decode($img['thumbCoords'], Json::FORCE_ARRAY);
				}
			}

			$imgs[$img['id']] = $i;
		}

		return $imgs;
	}
}
