<?php declare(strict_types = 1);

namespace GeneralServices\AdminModule\Components;

use Core\Model\TemplateReader\TemplateReaderControl;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseContainer;
use Core\Model\UI\Form\BaseForm;
use Core\Model\UI\Form\Controls\SelectInput;
use GeneralServices\Model\Entities\GeneralService;
use GeneralServices\Model\GeneralServices;
use GeneralServices\Model\Groups;
use Nette\Http\IResponse;
use Nette\Http\Request;
use Nette\Utils\ArrayHash;

class ServiceForm extends TemplateReaderControl
{
	/** @var GeneralService */
	public $service;

	/** @var GeneralServices */
	public $servicesService;

	/** @var Groups */
	public $groupsService;

	/** @var Request */
	protected $httpRequest;

	public function __construct(GeneralServices $services, Groups $groups)
	{
		$this->servicesService = $services;
		$this->groupsService   = $groups;
	}

	public function render()
	{
		$this->template->thisForm           = $this['form'];
		$this->template->componentStructure = $this->templateReader->getComponentStructure();
		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentForm()
	{
		$this->templateReader->setTemplatesDir($this->servicesService->getTemplateDir());

		$form   = $this->createForm();
		$groups = [];
		foreach ($this->groupsService->getEr()->findBy([], ['title' => 'ASC']) as $g) {
			$arr = [];
			$t   = $g;
			while ($t->parent) {
				$t     = $t->parent;
				$arr[] = $t->title;
			}
			$groups[$g->getId()] = ($arr ? implode(' -> ', $arr) . ' -> ' : '') . $g->title;
		}
		asort($groups);

		$templates = $this->servicesService->getTemplates();

		$form->addText('title', 'default.title')->setRequired()->setMaxLength(255);
		$form->addTextArea('introText', 'default.text');
		$form->addBool('isPublished', 'default.isPublished')->setDefaultValue(1);
		$form->addFilesManager('icon', 'default.image');
		$form->addCheckboxList('group', 'default.group', $groups)->setRequired();
		$form->addSelect('template', 'default.templates', $templates)->setPrompt('default.choose')->setRequired();
		$form->addComponent(new BaseContainer(), 'component');
		$form->addSubmit('submit', 'default.save');

		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values)
	{
		try {
			if ($this->service) {
				$service      = $this->service;
				$flashMessage = 'generalServices.serviceForm.edited';
			} else {
				$service      = new GeneralService($values->title);
				$flashMessage = 'generalServices.serviceForm.added';
			}

			$service->title       = $values->title;
			$service->isPublished = $values->isPublished;
			$service->groups      = $values->group ? array_map(function($id) {
				return $this->groupsService->get((int) $id);
			}, $values->group) : [];
			$service->template    = $values->template;
			$service->setTexts($form->getComponent('component')->getValues(true));

			$this->em->persist($service)->flush();
			$this->getPresenter()->flashMessageSuccess($flashMessage);
		} catch (\Exception $e) {
			$form->addError($e->getMessage());

			return false;
		}
	}

	public function setService($id)
	{
		$this->service = $this->servicesService->get($id);

		if ($this->service) {
			$form = $this['form'];
			$form->setDefaults([
				'title'       => $this->service->title,
				'isPublished' => $this->service->isPublished,
			]);


			if ($this->service->template && array_key_exists($this->service->template, $form['template']->getItems())) {
				$form['template']->setDefaultValue($this->service->template);

				if ($this->httpRequest->getQuery('do') != 'templatePageForm-loadInputs') {
					$this->templateReader->loadTemplateComponents($this['form']['component'], $this->httpRequest->getPost('template') ?: $this->service->template);
					$this->templateReader->setDefaults($this['form']['component'], $this->service);
				}
			}

			foreach ($this->service->groups as $g) {
				$d = [];
				if (array_key_exists($g->getId(), $this['form']['group']->getItems())) {
					$d[] = $g->getId();
				}

				$this['form']['group']->setDefaultValue($d);
			}
		} else
			$this->getPresenter()->error(null, IResponse::S404_NOT_FOUND);
	}

}
