<?php declare(strict_types = 1);

namespace Gls\AdminModule\Model\Subscribers;

use Contributte\Translation\Translator;
use Core\Model\Event\ControlEvent;
use EshopOrders\AdminModule\Components\Order\OrdersGrid;
use EshopOrders\AdminModule\Model\Event\OrderNavigationCallbackEvent;
use EshopOrders\AdminModule\Model\Event\OrdersNavigationEvent;
use EshopOrders\AdminModule\Model\Speditions;
use EshopOrders\Model\EshopOrdersConfig;
use Gls\Model\GlsConfig;
use Gls\Model\OrderApiService;
use Gls\Model\OrdersExported;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

class GlsOrdersSubscriber implements EventSubscriberInterface
{
	public function __construct(
		protected OrdersExported  $ordersExported,
		protected OrderApiService $apiService,
		protected Translator      $translator,
		protected Speditions      $speditionsService,
	)
	{
	}

	public static function getSubscribedEvents(): array
	{
		return [
			'eshopOrders.admin.ordersDefaultNavigation'         => 'ordersDefaultNavigation',
			'eshopOrders.admin.ordersDefaultNavigationCallback' => 'ordersDefaultNavigationCallback',
			OrdersGrid::class . '::onAttach'                    => ['ordersGridAttached', 255],
		];
	}

	public function ordersGridAttached(ControlEvent $event): void
	{
		/** @var OrdersGrid $control */
		$control = $event->control;

		$control->addExportCallback('gls', 'gls.ordersGrid.export', function(array $ids, $quantity) use ($control): void {
			$orders = $this->ordersExported->getOrdersNotExported($ids);
			if ($orders === []) {
				$control->presenter->flashMessageSuccess($this->translator->translate('gls.export.noOrders'));
				$control->presenter->redrawControl('flashes');

				return;
			}

			set_time_limit(120);
			$result = $this->apiService->sendOrders($orders, max(1, (int) $quantity));

			if ($result['error'] === 0) {
				$control->presenter->flashMessageSuccess($this->translator->translate('gls.export.success'));
			} else if ($result['error'] > 0 && $result['ok'] > 0) {
				$control->presenter
					->flashMessageWarning($this->translator->translate('gls.export.partial', null, $result));
			} else {
				$control->presenter->flashMessageDanger($this->translator->translate('gls.export.error'));
			}

			$control->presenter->redrawControl('flashes');
		});

		if (GlsConfig::load('allowGenerateLabel', false)) {
			$control->addExportCallback('glsLabels', 'gls.ordersGrid.labels', function(array $ids) use ($control): void {
				$orders = $this->ordersExported->getOrdersExported($ids);
				if ($orders === []) {
					$control->presenter->flashMessageSuccess($this->translator->translate('gls.labels.noOrders'));
					$control->presenter->redrawControl('flashes');

					return;
				}

				set_time_limit(120);
				$result = $this->apiService->generateLabels($orders);

				if ($result['error'] === 0) {
					foreach ($result['files'] as $file) {
						$arr                                         = [
							'name' => 'GLS labels',
							'url'  => $control->presenter->link(':Core:Admin:FileResponse:', base64_encode(serialize([
								'file'     => $file,
								'filename' => basename($file),
							]))),
						];
						$control->presenter->payload->fileRequests[] = $arr;
					}

					$control->presenter->flashMessageSuccess($this->translator->translate('gls.labels.success'));
				} else {
					$control->presenter->flashMessageDanger($this->translator->translate('gls.labels.error'));
				}

				$control->presenter->redrawControl('flashes');
			});
		}
	}

	public function ordersDefaultNavigation(OrdersNavigationEvent $event): void
	{
		if (EshopOrdersConfig::load('hideOrderGridNavExport')) {
			return;
		}

		$event->addNavigation(
			'gls.ordersNavigation.exportAll',
			['navigationCallback!', ['key' => 'glsExport']],
			'fas fa-truck-loading',
			'ajax'
		);
	}

	public function ordersDefaultNavigationCallback(OrderNavigationCallbackEvent $event): void
	{
		if ($event->key !== 'glsExport') {
			return;
		}

		$orders = $this->ordersExported->getOrdersNotExported();

		$ids = [];
		foreach ($orders as $k => $order) {
			$ids[] = $order->getOrder()->getId();
		}

		if ($ids !== []) {
			$allowed = $this->speditionsService->checkOrdersGridMenuExportOnlyInStatus($ids);

			foreach ($orders as $k => $order) {
				if (!in_array($order->getOrder()->getId(), $allowed)) {
					unset($orders[$k]);
				}
			}
		}

		if ($orders === []) {
			$event->presenter->flashMessageSuccess($this->translator->translate('gls.export.noOrders'));
			$event->presenter->redrawControl('flashes');

			return;
		}

		set_time_limit(120);
		$result = $this->apiService->sendOrders($orders);

		if ($result['error'] === 0) {
			$event->presenter->flashMessageSuccess($this->translator->translate('gls.export.success'));
		} else if ($result['error'] > 0 && $result['ok'] > 0) {
			$event->presenter->flashMessageWarning($this->translator->translate('gls.export.partial', null, $result));
		} else {
			$event->presenter->flashMessageDanger($this->translator->translate('gls.export.error'));
		}

		$event->presenter->redrawControl('flashes');
	}
}
