<?php declare(strict_types = 1);

namespace Gls\AdminModule\Model\Subscribers;

use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Event\ControlEvent;
use Core\Model\Event\CreateFormEvent;
use Core\Model\Event\Event;
use Core\Model\Event\EventDispatcher;
use Core\Model\Event\FormSuccessEvent;
use Core\Model\Event\GridFilterEvent;
use Core\Model\Event\SetFormDataEvent;
use Core\Model\UI\Form\BaseForm;
use Doctrine\Common\Collections\Criteria;
use EshopOrders\AdminModule\Components\Order\OrderForm;
use EshopOrders\AdminModule\Components\Order\OrdersGrid;
use EshopOrders\AdminModule\Components\Order\OrderSpeditionForm;
use EshopOrders\AdminModule\Components\Order\OrderStatusesGrid;
use EshopOrders\AdminModule\Model\Event\OrderStatusEvent;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderSpedition;
use EshopOrders\Model\Entities\OrderStatus;
use Exception;
use Gls\Model\Entities\GlsOrder;
use Gls\Model\Entities\GlsParcelNumber;
use Gls\Model\OrdersExported;
use Nette\Utils\Html;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Tracy\Debugger;

class OrderSubscriber implements EventSubscriberInterface
{
	/** @var GlsOrder[]|null */
	protected static ?array $allOrders = null;

	public function __construct(
		protected OrdersExported         $ordersExported,
		protected EntityManagerDecorator $em,
		protected EventDispatcher        $eventDispatcher,
	)
	{
	}

	public static function getSubscribedEvents(): array
	{
		return [
			OrderForm::class . '::onAttach'               => 'orderFormAttached',
			OrdersGrid::class . '::columnSpeditionRender' => 'columnSpeditionRender',
			OrdersGrid::class . '::filterPackageNumber'   => 'gridFilterPackageNumber',
			OrderSpeditionForm::class . '::createForm'    => 'orderSpeditionCreateForm',
			OrderSpeditionForm::class . '::formSuccess'   => 'orderSpeditionFormSuccess',
			OrderSpeditionForm::class . '::setOrder'      => 'orderSpeditionSetOrder',
			OrderStatusesGrid::class . '::beforeDelete'   => 'orderStatusDelete',
		];
	}

	public function orderFormAttached(ControlEvent $event): void
	{
		/** @var OrderForm $control */
		$control   = $event->control;
		$order     = $control->order;
		$spedition = $order->getSpedition() ? $order->getSpedition()->getSpedition() : null;

		if ($spedition && $spedition->getIdent() === 'gls') {
			$control->template->glsExport                   = $this->ordersExported->getOrdersExported([$order->getId()])[$order->getId()] ?? null;
			$control->template->speditionExportInfoTemplate = __DIR__ . '/exportInfoTemplate.latte';
		}
	}

	public function columnSpeditionRender(Event $event): void
	{
		$data           = $event->data;
		$speditionIdent = 'gls';
		$export         = null;

		/** @var Order $order */
		$order = $data['order'];

		/** @var Html $html */
		$html = $data['html'];

		if ($data['ordersBySpedition'][$speditionIdent] ?? null) {
			if (self::$allOrders === null) {
				self::$allOrders = $this->ordersExported->getOrdersExported(array_keys($data['ordersBySpedition'][$speditionIdent] ?? []), false);
			}
			$export = self::$allOrders[$order->getId()] ?? null;
		} else if ($order->getSpeditionIdent() === $speditionIdent) {
			$export = $this->ordersExported->getOrdersExported([$order->getId()]);
		}

		if ($export) {
			$h     = $html->addHtml(Html::el('div'));
			$urls  = $export->getTrackingUrls();
			$count = count($urls);
			$i     = 1;
			foreach ($urls as $number => $url) {
				$h->addHtml(Html::el('a', ['target' => '_blank'])
					->setAttribute('href', $url)
					->setText($number)
				);
				if ($i !== $count) {
					$h->addText(', ');
				}
				$i++;
			}
		}
	}

	public function gridFilterPackageNumber(GridFilterEvent $event): void
	{
		$ids = $this->ordersExported->findIdByPackageNumber($event->value);

		if ($ids !== []) {
			$event->criteria->orWhere(Criteria::expr()->in('o.id', $ids));
		}
	}

	public function orderSpeditionCreateForm(CreateFormEvent $event): void
	{
		/** @var OrderSpeditionForm $control */
		$control = $event->control;
		$form    = $event->form;

		$glsIds = [];
		foreach ($control->getSpeditions() as $sped) {
			if ($sped->getIdent() === 'gls') {
				$glsIds[] = $sped->getId();
			}
		}

		if ($glsIds === []) {
			return;
		}

		$container = $form->addContainer('gls');
		$container->addText('numberPackage', 'gls.entity.numberPackage')
			->setDescription('gls.entity.numberPackageDesc');
		$container->addBool('removeExport', 'gls.entity.removeExportStatus')
			->setDefaultValue(0);
		$container->addBool('updateDeliveryAddress', 'gls.entity.updateDeliveryAddress')
			->setDefaultValue(0);

		$form->getComponent('spedition')->addCondition(BaseForm::IS_IN, $glsIds)
			->toggle($form['gls']['numberPackage']->getHtmlId())
			->toggle($form['gls']['removeExport']->getHtmlId())
			->toggle($form['gls']['updateDeliveryAddress']->getHtmlId());
	}

	public function orderSpeditionFormSuccess(FormSuccessEvent $event): void
	{
		/** @var OrderSpedition $orderSpedition */
		$orderSpedition = $event->custom['entity'];
		/** @var OrderSpeditionForm $control */
		$control  = $event->control;
		$orderId  = $orderSpedition->getOrder()->getId();
		$values   = $event->values;
		$glsOrder = $this->ordersExported->getOrders(null, [$orderId])[$orderId] ?? null;

		$isGls = false;
		foreach ($control->getSpeditions() as $sped) {
			if ($sped->getIdent() === 'gls' && $sped->getId() == $values->spedition) {
				$isGls = true;
				break;
			}
		}

		try {
			if ($isGls) {
				if (!$glsOrder) {
					$glsOrder = new GlsOrder($orderSpedition->getOrder());
				}
				$numberPackages      = trim(str_replace(' ', '', (string) $values->gls->numberPackage)) ?: null;
				$numberPackages      = $numberPackages ? trim($numberPackages, ',') : null;
				$numberPackages      = $numberPackages && \str_contains($numberPackages, ',') ? array_map('trim', explode(',', $numberPackages)) : [];
				$numberPackagesCount = count($numberPackages);
				if ($numberPackagesCount <= 1) {

					$glsOrder->numberPackage = $numberPackages[0] ?? null;
					$glsOrder->associatedNumberPackages->clear();
				} else {

					$existsPackages            = $glsOrder->associatedNumberPackages->getKeys();
					$existsPackagesWithoutMain = array_slice($numberPackages, 1, count($numberPackages) - 1);
					$diffToAdd                 = array_diff($existsPackagesWithoutMain, $existsPackages);
					$diffToRemove              = array_diff($existsPackages, $existsPackagesWithoutMain);

					for ($i = 0; $i < $numberPackagesCount; $i++) {
						$numberPackage = $numberPackages[$i];
						if ($i === 0) {
							$glsOrder->numberPackage = $numberPackage;
						} else if (in_array($numberPackage, $diffToAdd)) {
							$glsOrder->associatedNumberPackages->add(new GlsParcelNumber($numberPackage, $glsOrder));
						}
					}

					foreach ($diffToRemove as $p) {
						$glsOrder->associatedNumberPackages->remove($p);
					}
				}
				if ($glsOrder->numberPackage && $glsOrder->getExported() === null) {
					$glsOrder->export();
				}
				if ($values->gls->removeExport === 1) {
					$glsOrder->resetExport();
				}
				$this->em->persist($glsOrder);
				if ($event->presenter && $values['gls']->updateDeliveryAddress) {
					$event->presenter->flashMessageWarning('gls.orderForm.updateDeliveryAddress');
					$event->presenter->redrawControl('flashes');
				}
				$this->eventDispatcher->dispatch(new Event(['order' => $glsOrder->getOrder(), 'numberPackages' => $numberPackages]), 'orderSubscriber.orderSpeditionFormSuccess');
			} else if ($glsOrder) {
				$this->em->remove($glsOrder);
			}
		} catch (Exception $e) {
			Debugger::log($e, 'orderSpeditionForm');
			$event->presenter->flashMessageDanger('gls.orderForm.speditionSaveError');
		}
	}

	public function orderSpeditionSetOrder(SetFormDataEvent $event): void
	{
		/** @var OrderSpedition $orderSpedition */
		$orderSpedition = $event->entity;
		$orderId        = $orderSpedition->getOrder()->getId();

		$glsOrder = $this->ordersExported->getOrders(null, [$orderId])[$orderId] ?? null;
		if (!$glsOrder) {
			return;
		}

		$event->form->getComponent('gls')->setDefaults([
			'numberPackage' => implode(', ', $glsOrder->getAllNumberPackages()),
		]);
	}

	public function orderStatusDelete(OrderStatusEvent $event): void
	{
		$orderStatus = $event->orderStatus;
		$order       = $orderStatus->getOrder();

		if ($orderStatus->getStatus()->getId() !== OrderStatus::STATUS_SPEDITION || ($order->getSpeditionIdent() && $order->getSpeditionIdent() !== 'gls')) {
			return;
		}

		$orderId  = $order->getId();
		$glsOrder = $this->ordersExported->getOrders(null, [$orderId])[$orderId] ?? null;

		if (!$glsOrder) {
			return;
		}

		$glsOrder->resetExport();
		$this->em->persist($glsOrder);
		$this->em->flush();
	}

}
