<?php declare(strict_types = 1);

namespace Gls\Model\Libs;

use Gls\Model\Exception\ParcelGeneration;
use Gls\Model\Exception\ParcelLabels;
use Gls\Model\Exception\ParcelDeletion;
use DOMDocument;
use GuzzleHttp\Exception\GuzzleException;
use Monolog\Handler\RotatingFileHandler;
use Monolog\Handler\SyslogHandler;
use Monolog\Logger;
use Nette\Utils\ArrayHash;
use nusoap_client;
use GuzzleHttp\Client;
use Symfony\Component\OptionsResolver\OptionsResolver;
use Respect\Validation\Validator as v;
use Respect\Validation\Exceptions\NestedValidationException;
use Gls\Model\Exception;

/**
 * https://github.com/schiggi/gls-cee-shipping-api
 */
class Api
{
	/**
     * @var string[]
     */
    protected array $services = ["T12", "PSS", "PRS", "XS", "SZL", "INS", "SBS", "DDS", "SDS", "SAT", "AOS", "24H", "EXW", "SM1", "SM2", "CS1", "TGS", "FDS", "FSS", "PSD", "DPV"];
	/**
     * @var string[]
     */
    protected array $label_size = ["A6", "A6_PP", "A6_ONA4", "A4_2x2", "A4_4x1", "T_85x85"];
	protected array $config;
	protected ?Logger $logger = null;
	/**
     * @var array<string, string>
     */
    protected array $urls = [
		'HU' => 'https://online.gls-czech.com/webservices/soap_server.php?wsdl&ver=18.09.12.01',
		'HU-TEST' => 'https://test.gls-czech.com/webservices/soap_server.php?wsdl&ver=15.04.18.01',
		'SK' => 'http://online.gls-slovakia.sk/webservices/soap_server.php?wsdl&ver=18.09.12.01',
		'CZ' => 'http://online.gls-czech.com/webservices/soap_server.php?wsdl&ver=18.09.12.01',
		'RO' => 'http://online.gls-romania.ro/webservices/soap_server.php?wsdl&ver=18.09.12.01',
		'SI' => 'http://connect.gls-slovenia.com/webservices/soap_server.php?wsdl&ver=18.09.12.01',
		'HR' => 'http://online.gls-croatia.com/webservices/soap_server.php?wsdl&ver=18.09.12.01',
	];

	/**
	 * @param array|ArrayHash $options
	 */
	public function __construct($options)
	{
		$this->config = $this->resolveOptions((array) $options);
		try {
			v::key('username', v::stringType()->notEmpty()->length(1,20))
				->key('password', v::stringType()->notEmpty()->length(1,20))
				->key('client_number', v::stringType()->notEmpty()->length(1,20))
				->key('country_code', v::stringType()->notEmpty()->in(array_keys($this->urls)))
				->key('label_paper_size', v::stringType()->notEmpty()->in(array_values($this->label_size)))
				->assert($this->config);
		} catch(NestedValidationException $exception) {
			echo $exception->getFullMessage();
		}

		if (!empty($this->config['log_dir'])) {
			$this->logger = $this->getLogger();
		}
	}

	/**
	 * Get required options for the GLS API to work
	 */
	protected function resolveOptions(array $opts): array
	{
		$resolver = new OptionsResolver;
		$resolver->setDefault('url', '');
		$resolver->setDefault('client_number', '');
		$resolver->setDefault('country_code', 'HU');
		$resolver->setDefault('label_paper_size', 'A4_2x2');
		$resolver->setDefault('log_dir', '');
		$resolver->setDefault('log_rotation_days', '7');
		$resolver->setDefault('log_syslog', false);
		$resolver->setDefault('log_logdna_key', false);
		$resolver->setDefault('log_msg_format', ['{method} {uri} HTTP/{version} {req_body}','RESPONSE: {code} - {res_body}',]);
		$resolver->setRequired(['url', 'username', 'password','client_number']);
		return $resolver->resolve($opts);
	}

	/**
	 * Get api url based on country code
	 */
	protected function getApiUrl(): string
	{
		return $this->urls[strtoupper((string) $this->config['country_code'])];
	}

	/**
     * Send parcels in batch
     *
     * @return array <pre> {
     *      [clientRef] => '123'
     *      [clientRef] => 'Error Description'
     * } </pre>
     * @throws Exception\ParcelGeneration
     * @param array<int, mixed> $parcel_data
     */
    public function getParcelNumbers(array $parcel_data): ?array
	{
		date_default_timezone_set("Europe/Budapest");
		$data = "<?xml version=\"1.0\" encoding=\"UTF-8\"?>";
		$data .="<DTU EmailAddress=\"" . $parcel_data[0]['SenderEmail'] . "\" Version=\"16.12.15.01\" Created =\"" . date(DATE_ATOM) . "\" RequestType=\"GlsApiRequest\" MethodName=\"prepareLabels\">";
		$data .= '<Shipments>';

		foreach ($parcel_data as $parcel) {
			// validate parcel data
			$this->validateParcelPrepare($parcel);

			$parcel['CodCurr'] ??= 'HUF';
			// the smallest fraction is 5 for COD amount
			$parcel['CodAmount'] = (float)$parcel['CodAmount'];
			$data .= "<Shipment SenderID=\"" . $this->config["client_number"] . "\" ExpSenderID=\"\" PickupDate=\"" . (isset($parcel["PickupDate"]) ? date(DATE_ATOM, strtotime($parcel["PickupDate"])) : date(DATE_ATOM)) . "\" ClientRef=\"" . $parcel['ClientRef'] . "\" CODAmount=\"" . $parcel['CodAmount'] . "\" CODCurr=\"" . $parcel['CodCurr'] . "\" CODRef=\"" . $parcel['CodRef'] . "\" PCount=\"" . ($parcel["Pcount"] ?? "1") . "\" Info=\"".($parcel['ConsigComment'] ?? "" ) . "\">";
			$data .= "<From Name=\"" . $parcel['SenderName'] . "\" Address=\"" . $parcel['SenderAddress'] . "\" ZipCode=\"" . $parcel['SenderZipcode'] . "\" City=\"" . $parcel['SenderCity'] . "\" CtrCode=\"" . $parcel['SenderCountry'] . "\" ContactName=\"" . $parcel['SenderContact'] . "\" ContactPhone=\"" . $parcel['SenderPhone'] . "\" EmailAddress=\"" . $parcel['SenderEmail'] . "\" />";
			$data .= "<To Name=\"" . $parcel['ConsigName'] . "\" Address=\"" . $parcel['ConsigAddress'] . "\" ZipCode=\"" . $parcel['ConsigZipcode'] . "\" City=\"" . $parcel['ConsigCity'] . "\" CtrCode=\"" . $parcel['ConsigCountry'] . "\" ContactName=\"" . $parcel['ConsigContact'] . " #" . $parcel["ClientRef"] . "\" ContactPhone=\"" . $parcel["ConsigPhone"] . "\" EmailAddress=\"" . $parcel["ConsigEmail"] . "\" />";
			if (!empty($parcel['Services'])) {
				$data .= '<Services>';
				foreach ($parcel['Services'] as $service_code => $service_parameter) {
					$data .= "<Service Code=\"" . $service_code . "\" >";
					$data .= '<Info>';
					$data .= "<ServiceInfo InfoType=\"INFO\" InfoData=\"" . $service_parameter . "\" />";
					$data .= '</Info>';
					$data .= '</Service>';
				}
				$data .= '</Services>';
			}
			$data .= "</Shipment>";
		}
		$data .= "</Shipments>";
		$data .= "</DTU>";

		$in = [
			"username" => $this->config["username"],
			"password" => $this->config["password"],
			"senderid" => $this->config["client_number"],
			"data" => base64_encode(gzencode($data,9))
		];

		$this->log('Request body encoded: ' . $data);

		try {
			$return = $this->requestNuSOAP('preparelabels_gzipped_xml', $in);
		} catch (\Exception $e) {
			throw new ParcelGeneration($e->getMessage());
		}

		$return = gzdecode($return);
		$this->log('Response body encoded: ' . $return);

		$doc = new DOMDocument;
		$doc->loadXML($return);
		if ($doc->getElementsByTagName("Status")->item(0)->nodeValue == "success") {
			$return_data = $doc->getElementsByTagName("Shipment");
			$parcel_result = [];
			for ($i = $return_data->length; --$i >= 0;) {
				if (isset($return_data->item($i)->getElementsByTagName("long")->item(0)->nodeValue)) {
					$parcel_result[$return_data->item($i)->getAttribute("ClientRef")] = $return_data->item($i)->getElementsByTagName("long")->item(0)->nodeValue;
				} elseif ($return_data->item($i)->getElementsByTagName("Status")->item(0)->nodeValue == "failed") {
					$parcel_result[$return_data->item($i)->getAttribute("ClientRef")] = $return_data->item($i)->getElementsByTagName("Status")->item(0)->getAttribute("ErrorDescription");
				} else {
					$parcel_result[$return_data->item($i)->getAttribute("ClientRef")] = $return_data->item($i)->getElementsByTagName("Status")->item(0)->getAttribute("ErrorCode");
				}
			}
			return $parcel_result;
		}

		return null;
	}

	/**
	 * get parcel labels
	 *
	 * @return array
	 * $parcel_result = [
	 *      'status'            => 'success'
	 *      'error_description' => 'Error Description'
	 *      'pdf'               => is streamed labels. Use echo and correct pdf header to display
	 * ]
	 * @throws Exception\ParcelLabels
	 */
	public function getParcelLabels(array $parcel_ids): array
	{

		date_default_timezone_set("Europe/Budapest");
		$data = "<?xml version=\"1.0\" encoding=\"UTF-8\"?>";
		$data .="<DTU EmailAddress=\"test@gls-czech.com\" Version=\"16.12.15.01\" Created=\"" . date(DATE_ATOM) . "\" RequestType=\"GlsApiRequest\" MethodName=\"printLabels\">";
		$data .= '<Shipments>';
		foreach ($parcel_ids as $parcel_id) {
			$data .= "<Shipment><PclIDs><long>";
			$data .= $parcel_id;
			$data .= "</long></PclIDs></Shipment>";
		}
		$data .= "</Shipments>";
		$data .= "</DTU>";

		$in = [
			"username" => $this->config["username"],
			"password" => $this->config["password"],
			"senderid" => $this->config["client_number"],
			"data" => base64_encode(gzencode($data,9)),
			"printertemplate"=>$this->config["label_paper_size"],
			"is_autoprint_pdfs"=>false
		];

		try {
			$return = $this->requestNuSOAP('getprintedlabels_gzipped_xml', $in);
		} catch (\Exception $e) {
			throw new ParcelLabels($e->getMessage());
		}

		$return = gzdecode($return);

		$doc = new DOMDocument;
		$doc->loadXML($return);
		$return_data = $doc->getElementsByTagName("Parcels");

		$parcel_result = [];

		$parcel_result['status']            = $doc->getElementsByTagName("Status")->item(0)->nodeValue;
		$parcel_result['error_description'] = $doc->getElementsByTagName("Status")->item(0)->getAttribute("ErrorDescription");
		if (isset($return_data->item(0)->getElementsByTagName("Label")->item(0)->nodeValue)) {
			$parcel_result['pdf']           = base64_decode($return_data->item(0)->getElementsByTagName("Label")->item(0)->nodeValue);
		} else {
			$parcel_result['pdf'] = '';
		}

		// Logging
		$this->log('Request body encoded: ' . $data);
		$this->log('Response body status: ' . $parcel_result['status']);
		$this->log('Response body error description: ' . $parcel_result['error_description']);

		return $parcel_result;
	}

	/**
	 * Get parcel status
	 *
	 * @return int|false
	 */
	public function getParcelStatus(string $parcelNumber)
	{
		$statuses = $this->getParcelStatusList($parcelNumber);
		$delivery_code = $statuses[0]['StCode'];

		return isset($delivery_code) ? (int) $delivery_code : false;
	}

	/**
	 * @return array|false
	 * @throws GuzzleException
	 */
	public function getParcelStatusList(string $parcelNumber)
	{
		$config_array = [
			'verify' => false,
			'debug' => false
		];
		$client = new Client($config_array);
		$response = $client->request("GET", $this->getTrackingUrlXml($parcelNumber));
		$xml = simplexml_load_string((string) $response->getBody());

		try {
			if ($xml !== FALSE) {
				$arr = [];
				foreach ($xml->Parcel->Statuses->Status as $status) {
					$arr[] = ((array) $status)['@attributes'];
				}

				return $arr;
			}

			throw new Exception('Tracking code wasn`t registered or error occured!');
		} catch (\Exception) {
		}

		return false;
	}

	public function getTrackingUrl(string $parcelNumber, string $language = 'en'): string
	{
		return "http://online.gls-czech.com/tt_page.php?tt_value=$parcelNumber&lng=$language";
	}

	public function getTrackingUrlXml(string $parcelNumber): string
	{
		return "http://online.gls-czech.com/tt_page_xml.php?pclid=$parcelNumber";
	}

	/**
	 * Delete parcels in batch
	 *
	 * $parcel_result = [
	 *      $parcel_id            => $result_msg
	 * ]
	 * @throws Exception\ParcelDeletion
	 */
	public function deleteParcels(array $parcel_ids): array
	{
		date_default_timezone_set("Europe/Budapest");
		$data = "<?xml version=\"1.0\" encoding=\"UTF-8\"?>";
		$data .="<DTU EmailAddress=\"test@gls-czech.com\" Version=\"16.12.15.01\" Created=\"" . date(DATE_ATOM) . "\" RequestType=\"GlsApiRequest\" MethodName=\"deleteLabels\">";
		$data .= '<Shipments>';
		foreach ($parcel_ids as $parcel_id) {
			$data .= "<Shipment><PclIDs><long>";
			$data .= $parcel_id;
			$data .= "</long></PclIDs></Shipment>";
		}
		$data .= "</Shipments>";
		$data .= "</DTU>";

		$this->log('Request body encoded: ' . $data);

		$in = [
			"username" => $this->config["username"],
			"password" => $this->config["password"],
			"senderid" => $this->config["client_number"],
			"data" => base64_encode(gzencode($data,9))
		];

		try {
			$return = $this->requestNuSOAP('deletelabels_gzipped_xml', $in);
		} catch (\Exception $e) {
			throw new ParcelDeletion($e->getMessage());
		}

		$return = gzdecode($return);
		$this->log('Response body encoded: ' . $return);

		$parcel_result = [];
		if (!in_array($return, ['', '0', false], true)) {
			$doc = new DOMDocument;
			$doc->loadXML($return);
			$return_data = $doc->getElementsByTagName("Shipment");
			for ($i = $return_data->length; --$i >= 0;) {
				if ($return_data->item($i)->getElementsByTagName("Status")->item(0)->nodeValue == "success") {
					$parcel_result[$return_data->item($i)->getElementsByTagName("Parcel")->item(0)->getAttribute("PclId")] = $return_data->item($i)->getElementsByTagName("Status")->item(0)->nodeValue;
				} elseif ($return_data->item($i)->getElementsByTagName("Status")->item(0)->nodeValue == "failed") {
					$parcel_result[$return_data->item($i)->getElementsByTagName("Parcel")->item(0)->getAttribute("PclId")] = $return_data->item($i)->getElementsByTagName("Status")->item(0)->getAttribute("ErrorDescription");
				} else {
					$parcel_result[$return_data->item($i)->getElementsByTagName("Parcel")->item(0)->getAttribute("PclId")] = $return_data->item($i)->getElementsByTagName("Status")->item(0)->getAttribute("ErrorCode");
				}
			}
		}

		return $parcel_result;
	}


	/**
	 * @param string $method
	 * @param array $data
	 * @return mixed
	 */
	protected function requestNuSOAP($method, $data = []) {
		$client = new nusoap_client($this->getApiUrl(), 'wsdl');

		$error  = $client->getError();
		if ($error) {
			$this->logger->error($error);
		}

		return $client->call($method, $data);
	}

	protected function request(string $uri, array $data = [], string $method = 'GET', array $headers = []): string
	{
		$config_array = [
			'verify' => false,
			'debug' => false
		];
		$client = new Client($config_array);
		$response = $client->request($method, $uri, ['query' => $data, 'headers' =>$headers]);

		return $response->getBody()->getContents();
	}

	/**
	 *	Logger functionality: Creates a log file for each day with all requests and responses
	 */
	private function getLogger(): Logger
	{
		if (!$this->logger instanceof Logger) {
			$this->logger = new Logger('api-gls-consumer');
			$this->logger->pushHandler(
				new RotatingFileHandler( $this->config['log_dir'] . 'api-gls-consumer.log', (int) $this->config['log_rotation_days'])
			);
		}
		if (!empty($this->config['log_syslog'])) {
			$this->logger->pushHandler(
				new SyslogHandler('api-gls-consumer')
			);
		}

		return $this->logger;
	}

	private function log(string $msg): void
	{
		if ($this->logger instanceof Logger) {
			$this->logger->info($msg);
		}
	}

	/**
	 *	Validation functions
	 */
	private function validateParcelPrepare(array $data): void
	{
		try {
			v::key('Services', v::ArrayType())
				// check date format for pickup date
				->key('PickupDate', v::date()->notEmpty())
				->assert($data);
			v::optional(
			// check cod ref length
				v::key('CodRef', v::stringType()->length(0,512))
					//check phone number for receiver (international format)
					->keyNested('Services.FSS',v::phone())
			)->assert($data);
		} catch(NestedValidationException $exception) {
			echo $exception->getFullMessage();
		}
	}
}
